; This model simulates the formation of photochemical smog over time. 

globals [
  earth-top      ;; y-coordinate that splits earth from atmosphere
  timeofday      ;; time of day in hours (0-24)
  sunlight       ;; level of sunlight (0-100)
  p-rate         ;; adjuster for emission rate of pollutants (0-1)
  NO2-rate       ;; rate of NO2-splitting reaction
  O3-rate        ;; rate of O3-formation reaction
  sunrise        ;; time at which sun rises
  sunset ]       ;; time at which sun sets

patches-own [
  patch-type ]   ;; 0 = earth, 1 = buildings, 2 = sky

breed [VOC]   ;; volatile organic chemicals
breed [NO2]   ;; nitrogen dioxide
breed [NO]    ;; nitrogen oxide
breed [O1]    ;; oxygen radicals
breed [O2]    ;; oxygen
breed [O3]    ;; ozone
breed [Ox]    ;; photochemical oxidants


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;; SETUP PROCEDURES ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to Setup
  clear-all
  set-default-shape VOC "dot"   ;; set default shapes for all turtle breeds
  set-default-shape NO2 "dot"
  set-default-shape O3 "dot"
  set-default-shape NO "dot"
  set-default-shape O1 "dot"
  set-default-shape O2 "dot"  
  set-default-shape Ox "dot"  
  setup-world
  set sunrise 420
  set sunset 1140
  set timeofday 0                ;; reset clock 
  reset-ticks                    ;; reset tick counter
end
  
to setup-world
  set earth-top -12
  set sunlight sunlight-intensity
  ask patches with [             ;; create grass
    pycor < earth-top ] [
    set patch-type 0
    set pcolor 64 ]
  ask patches with [
    pycor >= earth-top ] [        ;; create sky
    set patch-type 2 
    set pcolor scale-color sky sunlight 0 100 ]
  ask patches with [             ;; create buildings
    pycor >= earth-top and
    pycor <= earth-top + 1 and
    pxcor >= -8 and
    pxcor <= 8 ]
    [ set patch-type 1 
    set pcolor 3 ]
  ask patches with [
    pycor >= earth-top and
    pycor <= earth-top + 4 and
    pxcor >= 2 and
    pxcor <= 3 ]
    [ set patch-type 1 
    set pcolor 3 ]
  ask patches with [
    pycor >= earth-top and
    pycor <= earth-top + 6 and
    pxcor >= -1 and
    pxcor <= 1 ]
    [ set patch-type 1 
    set pcolor 3 ]    
  ask patches with [
    pycor >= earth-top and
    pycor <= earth-top + 2 and
    pxcor >= -5 and
    pxcor <= -3 ]
    [ set patch-type 1 
    set pcolor 3 ]
  ask patches with [
    pycor >= earth-top and
    pycor <= earth-top + 4 and
    pxcor > -4 and
    pxcor <= -3 ]
    [ set patch-type 1 
    set pcolor 3 ]
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;; RUNTIME PROCEDURES ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to Go
  ifelse timeofday < 1440 [        ;; time progresses, one minute per tick
    set timeofday timeofday + 1 ][
    set timeofday 0]  
  update-sunlight
  ifelse pollutants-cycle [        ;; add primary pollutants to atmosphere
    cycle-pollutants]
    [add-pollutants]  
  ifelse atmosphere-inversion [    ;; move the pollutants around 
    run-pollutants-inversion]
    [run-pollutants]             
  run-reactions                    ;; perform chemical reactions on primary pollutants 
  tick
end



to update-sunlight
  ifelse day-night-cycle [    
    if timeofday < sunrise or timeofday >= sunset [      ;; night
      set sunlight 0]
    if timeofday >= sunrise and timeofday < sunset [     ;; day
      set sunlight sunlight-intensity ]
  ]
    [set sunlight sunlight-intensity]
  ask patches with [
    patch-type = 2 ] [ 
    set pcolor scale-color sky sunlight 0 100 ]             ;; color sky according to sunlight level  
end



to cycle-pollutants            ;;;;;;;;;; Add pollutants to the atmosphere in a cycle ;;;;;;;;;
  if timeofday < sunrise or timeofday >= sunset [            ;; night (no pollution)
    set p-rate 0]
  if timeofday >= sunrise and timeofday < sunset  [           ;; day (max pollution rate)
    set p-rate 1 ]
  create-NO2 p-rate * nitrogen-dioxide [
    set color brown
    setxy random-xcor earth-top + random-float max-pycor]
  create-VOC p-rate * volatile-organic-compounds [
    set color gray
    setxy random-xcor earth-top + random-float max-pycor]   
end



to add-pollutants             ;;;;;;;;;; Add pollutants to the atmosphere (no cycle) ;;;;;;;;;
  create-NO2 nitrogen-dioxide [
    set color brown
    setxy 0 -6
    set heading -90 + random-float 180 ]
  create-VOC volatile-organic-compounds [
    set color gray
    setxy 0 -6 
    set heading -90 + random-float 180 ] 
end



to run-pollutants-inversion    ;;;;;;;;;; Make pollutants move around, with atmospheric inversion ;;;;;;;;;
  ask turtles [
    let dist 0.5 * random-float 1
    ifelse can-move? dist
      [ fd dist ]
      [ set heading heading + 180 
        fd dist ]                     ;; if turtle hits the edge of the world, it turns around
    if ycor <= earth-top 
      [ set heading random-float 80 ] ;; if turtle hits ground, return it to atmosphere
  ]
end




to run-pollutants              ;;;;;;;;;; Make pollutants move around (no inversion) ;;;;;;;;;
  ask turtles [
    let dist 0.5 * random-float 1
    ifelse can-move? dist
      [ fd dist ]
      [ die ]                  ;; if turtle hits the edge of the world, it dies
    if ycor <= earth-top 
      [ set heading random 80 ] ;; if turtle hits ground, return it to atmosphere
  ]
end





to run-reactions                 ;;;;;;;;;;; Chemical reactions WITHOUT O2 turtles ;;;;;;;;;;
  set NO2-rate sunlight                   ;; set reaction rate for splitting NO2
  set O3-rate sunlight                    ;; set reaction rate for forming O3
  ask NO2 [                               ;; (1) SPLIT NO2 
    if random-float 200 < NO2-rate [        ;; roll dice to see if there'e enough energy for reaction
      set breed O1                            ;; if there is, convert the NO2 turtle to O1
      set color yellow
      hatch-NO 1 [                            ;; and create a new turtle that is NO
        set color orange 
        set heading random-float 360 ]]      
  ]
  ask O1 [                                ;; (2) CREATE OZONE
    if random-float 200 < O3-rate [         ;; roll dice to see if there's enough energy for reaction
        set breed O3                          ;; if there is, convert O1 turtle to O3
        set color violet
        set heading random-float 360 ]       
  ]
  ask NO [                                ;; (3) DESTROY OZONE
    if any? O3-here [                       ;; see if there are any O3 in the patch
      set breed NO2                           ;; if there are, convert NO turtle to NO2
      set color brown
      set heading random-float 360 ]  
    let reactant one-of O3-here               ;; destroy used up O3 turtle 
    if reactant != nobody [
      ask reactant [ die ]]
  ]  
  ask VOC [                               ;; (4) CREATE OXIDANTS
    if any? NO-here [                       ;; see if any NO are in the patch
      set breed Ox                            ;; if there are, convert VOC turtle to Ox
      set color red
      set heading random-float 360 ]     
    let reactant one-of NO-here               ;; destroy used up NO turtle
    if reactant != nobody [
      ask reactant [die]]
  ] 
end





    
@#$#@#$#@
GRAPHICS-WINDOW
210
10
727
392
19
13
13.0
1
10
1
1
1
0
0
0
1
-19
19
-13
13
0
0
1
ticks
30.0

BUTTON
13
10
79
43
NIL
Setup\\n
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
93
10
156
43
NIL
Go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
4
59
204
92
nitrogen-dioxide
nitrogen-dioxide
0
5
2
1
1
NIL
HORIZONTAL

SLIDER
4
98
204
131
volatile-organic-compounds
volatile-organic-compounds
0
5
2
1
1
NIL
HORIZONTAL

SWITCH
4
254
199
287
atmosphere-inversion
atmosphere-inversion
1
1
-1000

PLOT
377
394
726
579
Molecule Counts
Time
Molecules
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"VOC" 1.0 0 -7500403 true "" "plot count VOC"
"NO2" 1.0 0 -6459832 true "" "plot count NO2"
"NO" 1.0 0 -955883 true "" "plot count NO"
"O1" 1.0 0 -1184463 true "" "plot count O1"
"O3" 1.0 0 -8630108 true "" "plot count O3"
"Ox" 1.0 0 -2674135 true "" "plot count Ox"

SWITCH
4
176
199
209
day-night-cycle
day-night-cycle
1
1
-1000

SWITCH
4
215
199
248
pollutants-cycle
pollutants-cycle
1
1
-1000

SLIDER
4
137
204
170
sunlight-intensity
sunlight-intensity
0
100
100
1
1
NIL
HORIZONTAL

PLOT
5
394
376
579
Total smog
Time
Smog molecules
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"smog" 1.0 0 -5825686 true "" "plot count O3 + count Ox"
"primary pollutants" 1.0 0 -10402772 true "" "plot count NO2 + count VOC"

@#$#@#$#@
## WHAT IS IT?

This model simulates the formation of photochemical smog over time. Nitrogen dioxide (NO2) and volatile organic compounds (VOC) are primary pollutants that are released into the atmosphere as a result of fossil fuel combustion. In the presence of oxygen (O2) and sunlight, they undergo chemical reactions that form ozone (O3) and photochemical oxidants (Ox), which are the principal components of smog.

## HOW IT WORKS

The primary pollutants, NO2 and VOC, are emitted at rates designated by the user. When nitrogen oxide (NO2) is the only primary pollutant present, the following reactions occur:

  (1)  NO2 + sunlight = NO + O  (nitrogen dioxide breaks down)
  (2)  O + O2 + sulight = O3    (ozone is created)
  (3)  NO + O3 = O2 + NO2       (ozone is destroyed)

When VOC is present in addition to NO2, the following reaction replaces reaction (3) above:

  (4)  NO + VOC = Ox  (photochemical oxidants are created)

So, the presence of VOC causes a build-up of ozone in the atmosphere and leads to the formation of photochemical oxidants, which together are what we call smog. The chemical reaction rates of (1) and (2) are dependent on sunlight, so higher levels of sunlight cause this reaction to occur faster. When there is no sunlight, these reactions do not occur.

## HOW TO USE IT

You can adjust several variables that will affect the behavior of the model:

  "nitrogen-dioxide" - the number of NO2 molecules created during each tick
  "volatile-organic-compounds" - the number of VOC molecules created during each tick
  "sunlight-intensity" - the level of sunlight (0 - 100)

You can also choose the conditions under which the model will run:

  When the "day-night-cycle" switch is OFF, the sunlight level will remain the same over time, at the level that the user selected. When the switch is ON, the level of sunlight will change over time, going from 0 at "night" to the user's selected setting during the "day."

  When the "pollutants-cycle" switch is OFF, the emissions rates for NO2 and VOC will remain the same over time, at the level that the user selected. When the switch is ON, the emissions rate will change over time going from 0 at "night" to the user's selected setting during the "day." This reflects the fact that most emission-creating activities occur during daylight hours.

  When the "atmosphere-inversion" switch is OFF, molecules that reach the edge of the screen disappear. This simulates a situation in which normal air flow causes dispersion of pollutants away from their source. When the switch is ON, molecules that reach the edge of the screen bounce back into the screen area. This reflects a situation in which a temperature inversion in the atmosphere prevents pollutants from dispersing. 

When you are ready to run the model, click the "Setup" button, then click "Go."

## THINGS TO NOTICE

The model assumes that the supply of oxygen (O2) is infinite. This is a realistic assumption for the reactions occuring in this model. Oxygen makes up about 21% of the atmosphere--that's 210,000 parts per million (ppm). Dangerous levels of smog are on the order of 0.4 ppm. In order to make the number of oxygen molecules in the model be in realistic proportion to the pollutant molecules, there would have to be millions of them, which is not feasible. So instead of using O2 turtles, the model assumes that O2 is always available.

## THINGS TO TRY

Try each of the switches separately, then try them together. Compare the results.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

orbit 3
true
0
Circle -7500403 true true 116 11 67
Circle -7500403 true true 26 176 67
Circle -7500403 true true 206 176 67
Circle -7500403 false true 45 45 210

orbit 5
true
0
Circle -7500403 true true 116 11 67
Circle -7500403 true true 13 89 67
Circle -7500403 true true 178 206 67
Circle -7500403 true true 53 204 67
Circle -7500403 true true 220 91 67
Circle -7500403 false true 45 45 210

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.0.2
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 1.0 0.0
0.0 1 1.0 0.0
0.2 0 1.0 0.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
