breed [ birds bird ]
breed [ dead-birds dead-bird ]


birds-own [
  flockmates         ;; agentset of nearby turtles
  nearest-neighbor   ;; closest one of our flockmates
  velocity           ;; vector with x y and z components it is determined by the previous velocity
                     ;; and the current acceleration at each step
  acceleration       ;; vector with x y and z components, determined by the six urges
]

;;
;; Setup Procedures
;;
to setup
  ca
  setxyz 0 -30 0
  create-birds population
    [
      place-bird
      set size 0.75
      set velocity [ 0 0 0 ]
      set acceleration [ 0 0 0 ]
      set color yellow + 2 - random-float 6
    ]
  ask turtle 0 [ set color red ]

  ;; checkered floor
  ask patches
    with [ pzcor = min-pzcor ]
      [ set pcolor 3 + green + ( pxcor + pycor ) mod 2 ]
end

to place-bird ;; turtle procedure
  setxyz random-xcor random-ycor random-zcor
  ;; make sure that the birds are not on the floor or inside the
  ;; any obstacles that have been created.
  while [ pcolor != black ]
  [ setxyz random-xcor random-ycor random-zcor ]
end

;;
;; Runtime Procedures
;;
to go
  ask birds
  [ flock
    ;; if we've hit something
    ;; we're a goner
    if pcolor != black
    [ hatch-dead-birds 1
      [ bk 1 ]
      die ]
  ]
  ;; ask the dead ones to continue falling
  ask dead-birds
    [ fall ]
  tick
end

to flock  ;; bird procedure
  ;; look for birds in my vicinity
  find-flockmates

  ;; acceleration at each step is determined
  ;; entirely by the urges
  set acceleration (list 0 0 0)

  add-urge world-center-urge world-center-constant
  add-urge wander-urge wander-constant
  add-urge avoid-obstacle-urge avoidance-constant

  ;; if I'm not in a flock ignore the flock related
  ;; urges
  if count flockmates > 0
  [ add-urge spacing-urge spacing-constant
    add-urge center-urge center-constant
    add-urge velocity-urge velocity-constant ]

  ;; keep the acceleration within the accepted range
  if magnitude acceleration > max-acceleration
  [ set acceleration
    (scale
        max-acceleration
        normalize acceleration) ]

  ;; the new velocity of the bird is sum of the acceleration
  ;; and the old velocity.
  set velocity (add velocity acceleration)

  ;; keep the velocity within the accepted range
  if magnitude velocity > max-velocity
  [ set velocity
    (scale
        max-velocity
        normalize velocity) ]

  let nxcor xcor + ( item 0 velocity )
  let nycor ycor + ( item 1 velocity )
  let nzcor zcor + ( item 2 velocity )
  facexyz nxcor nycor nzcor
  fd 0.1 * magnitude velocity
end

to add-urge [urge factor] ;; bird procedure
  set acceleration add acceleration scale factor normalize urge
end

to-report world-center-urge  ;; bird reporter
  ifelse distancexyz 0 0 0 > 10
  [ report (list (- xcor) (- ycor) (- zcor) ) ]
  [ report (list 0 0 0) ]
end

to-report center-urge ;; bird reporter
  ;; report the average distance from my flockmates
  ;; in each direction
  if count flockmates = 0 or center-constant = 0
  [ report (list 0 0 0) ]
  report
    (map
      [ ?2 - ?1 ]
      (list xcor ycor zcor)
      (list
        mean [ xcor ] of flockmates
        mean [ ycor ] of flockmates
        mean [ zcor ] of flockmates ) )
end

to-report velocity-urge ;; bird reporter
  ;; report the average difference in velocity
  ;; from my flock mates
  if count flockmates = 0 or velocity-constant = 0
  [ report (list 0 0 0) ]
  report
    ( map
      [ ?1 - ?2 ]
      (list
        mean [ item 0 velocity ] of flockmates
        mean [ item 1 velocity ] of flockmates
        mean [ item 2 velocity ] of flockmates )
      velocity )

end

to-report wander-urge ;; turtle reporter
  ;; report 3 random numbers between -1 and 1
  report n-values 3 [ (random-float 2) - 1 ]
end

to-report spacing-urge ;; turtle reporter
  let urge [ 0 0 0 ]
  ;; report the sum of the distances to birds
  ;; in my flock that are closer to me than
  ;; cruise-distance
  ask flockmates with [ distance myself < cruise-distance ] [
    set urge
      add
        urge
        (subtract
          (list [xcor] of myself [ycor] of myself [zcor] of myself)
          (list xcor ycor zcor))
  ]
  ;; when vertical spacing is not on ignore the
  ;; z component of the spacing urge.
  if not vertical-spacing? [ set urge lput 0 but-last urge ]
  report urge
end

to-report avoid-obstacle-urge ;; turtle reporter
  let urge (list 0 0 0)
  if avoidance-constant = 0 [ report urge ]
  ;; report the sum of the distances from
  ;; any patches that are obstacles
  ;; in each direction
  ask patches in-radius vision with [ pcolor != black ]
  [ set urge
      add
        urge
        subtract
          (list [xcor] of myself [ycor] of myself [zcor] of myself)
          (list pxcor pycor pzcor)
  ]
  report urge
end

to find-flockmates  ;; turtle reporter
  ;; latch on to the nearby birds
  set flockmates other birds in-radius vision
end

to fall ;; dead-bird procedure
  if zcor > min-pzcor + 1
    [ set zcor zcor - 0.5 ]
end

;;
;; world building
;;
to build-cubes
  ask n-of 20 patches
  [ set pcolor blue + random 3
    ask neighbors [ set pcolor blue + random 3 ] ]
end

to build-wall
  let x max-pxcor - random world-width
  let ydir 1 - random 2
  if ydir = 0 [ set ydir -1 ]
  let height random max-pxcor
  ask patches
    with [ pxcor = x and ydir * pycor < 0 and pzcor < height ]
    [ set pcolor gray + random 2 ]
end

;;
;; vector operations
;;
to-report add [ v1 v2 ]
  report (map [ ?1 + ?2 ] v1 v2)
end

to-report subtract [ v1 v2 ]
  report (map [ ?1 - ?2 ] v1 v2)
end

to-report scale [ scalar vector ]
  report map [ scalar * ? ] vector
end

to-report magnitude [ vector ]
  report sqrt sum map [ ? * ? ] vector
end

to-report normalize [ vector ]
  let m magnitude vector
  if m = 0 [ report vector ]
  report map [ ? / m ] vector
end


; Copyright 2005 Uri Wilensky. All rights reserved.
; The full copyright notice is in the Information tab.
@#$#@#$#@
GRAPHICS-WINDOW
0
0
260
281
12
12
10.0
1
10
1
1
1
0
1
1
1
-12
12
-12
12
-12
12
1
1
1
ticks

BUTTON
107
67
194
100
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

BUTTON
197
67
288
100
go
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL

SLIDER
94
32
309
65
population
population
0
200
60
1
1
NIL
HORIZONTAL

SLIDER
111
233
283
266
vision
vision
0
10
1.5
0.1
1
NIL
HORIZONTAL

SLIDER
186
405
358
438
wander-constant
wander-constant
0
20
4
1
1
NIL
HORIZONTAL

SLIDER
186
370
358
403
world-center-constant
world-center-constant
0
20
10
1
1
NIL
HORIZONTAL

SLIDER
13
370
185
403
center-constant
center-constant
0
20
8
1
1
NIL
HORIZONTAL

SLIDER
13
405
185
438
velocity-constant
velocity-constant
0
20
6
1
1
NIL
HORIZONTAL

SLIDER
190
302
362
335
spacing-constant
spacing-constant
0
20
14
1
1
NIL
HORIZONTAL

SLIDER
17
268
189
301
max-velocity
max-velocity
0
20
4
1
1
NIL
HORIZONTAL

SLIDER
190
268
362
301
max-acceleration
max-acceleration
0
20
1
1
1
NIL
HORIZONTAL

SLIDER
17
302
189
335
cruise-distance
cruise-distance
0
5
1
0.1
1
NIL
HORIZONTAL

BUTTON
133
193
250
226
NIL
follow turtle 0
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

SWITCH
105
336
277
369
vertical-spacing?
vertical-spacing?
0
1
-1000

SLIDER
103
439
275
472
avoidance-constant
avoidance-constant
0
100
50
1
1
NIL
HORIZONTAL

MONITOR
132
140
253
185
NIL
count dead-birds
3
1
11

BUTTON
96
102
194
135
NIL
build-cubes
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

BUTTON
197
102
295
135
NIL
build-wall
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

@#$#@#$#@
WHAT IS IT?
-----------
This is a vector-based 3D flocking model, based on Jon Klein's implementation of Craig Reynolds' Boids algorithm. Each bird is influenced by a series of urges. By assigning different weights to each urge, the birds exhibit different flocking behaviors.


HOW IT WORKS
------------
Each bird has vectors for velocity and acceleration, that is, a velocity and acceleration in the x, y, and z directions.  Each of these vectors is influenced by six urges, to be near the center of the flock, to have the same velocity as the rest of the flock, to be keep spacing correct, to avoid colliding with obstacles, to be near to the center of the world and to wander throughout the world.  Each of these factors affects the resulting velocity and acceleration; however, the sliders in the interface weight the amount that each has an effect.


HOW TO USE IT
-------------
Choose the number of birds with the POPULATION slider. Press SETUP to create the birds. Press GO to start the simulation.

BUILD-CUBES randomly places floating-ball obstacles throughout the world. BUILD-WALL creates a randomly placed wall.

VISION is the distance that a bird can see around it. MAX-VELOCITY is the fastest a bird can go and MAX-ACCELERATION is the fastest a bird can accelerate. CRUISE-DISTANCE is the minimum distance from a nearby bird for a bird to feel comfortable.

A bird's movement is determined by the set of urges acting on that bird:

CENTER-CONSTANT - urge to move towards the center of its flock
VELOCITY-CONSTANT - urge to align its velocity with the velocity of her flockmates
SPACING-CONSTANT - urge to be no closer than CRUISE-DISTANCE from other birds
AVOIDANCE-CONSTANT - urge to avoid colliding with obstacles
WORLD-CENTER-CONSTANT - urge to avoid the edges of the world
WANDER-CONSTANT - urge to move in a random way

The urge-constant sliders control the weight of each urge in determining the birds' behavior.

If the VERTICAL-SPACING? switch is off the spacing urge only operates horizontally.

If a bird hits an obstacle, the bird dies. The COUNT DEAD-BIRDS monitor shows the number of dead birds.


THINGS TO NOTICE
----------------
Compare how long it takes the birds to form flocks in this model to how long it takes in the regular 3D Flocking model.

Given the default settings the flocks travel in a circular pattern.


THINGS TO TRY
-------------
Turn VERTICAL-SPACING? off and notice how it changes the shape of the flocks.

Try changing the different constants so that the birds become better or worse at avoiding obstacles.


EXTENDING THE MODEL
-------------------
Try adding different urges to the model; also try making different types of obstacles for birds to avoid.

Create different breeds of birds that behave differently from each other and see how they interact.


NETLOGO FEATURES
----------------
This model uses lists and map fairly extensively to represent and manipulate vectors.


CREDITS AND REFERENCES
----------------------
This model is based on Jon Klein's "Swarm" demo for breve, (see http://www.spiderland.org/breve/) which was inspired by Craig Reynolds classic Boids flocking algorithm.


HOW TO CITE
-----------
If you mention this model in an academic publication, we ask that you include these citations for the model itself and for the NetLogo software:
- Wilensky, U. (2005).  NetLogo Flocking 3D Alternate model.  http://ccl.northwestern.edu/netlogo/models/Flocking3DAlternate.  Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.
- Wilensky, U. (1999). NetLogo. http://ccl.northwestern.edu/netlogo/. Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.

In other publications, please use:
- Copyright 2005 Uri Wilensky. All rights reserved. See http://ccl.northwestern.edu/netlogo/models/Flocking3DAlternate for terms of use.


COPYRIGHT NOTICE
----------------
Copyright 2005 Uri Wilensky. All rights reserved.

Permission to use, modify or redistribute this model is hereby granted, provided that both of the following requirements are followed:
a) this copyright notice is included.
b) this model will not be redistributed for profit without permission from Uri Wilensky. Contact Uri Wilensky for appropriate licenses for redistribution for profit.

@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

link
true
0
Line -7500403 true 150 0 150 300

link direction
true
0
Line -7500403 true 150 150 30 225
Line -7500403 true 150 150 270 225

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 3D 4.1pre7
@#$#@#$#@
setup
watch turtle 0
repeat 500 [ go ]
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
