globals
[
  tick-delta                         ;; how much we advance the tick counter this time through
  min-tick-delta                     ;; the smallest tick-delta is allowed to be
  init-avg-speed init-avg-energy     ;; initial averages
  collision-times                    ;; a list that of times of pending collisions

  ;; averages from the end of the last tick
  avg-speed avg-energy
  avg-speed-magenta
  avg-speed-cyan
  avg-energy-magenta
  avg-energy-cyan

  open?                              ;; is there an opening in the divider?
]

breed [ dividers divider ]
breed [ walls wall ]

breed [ particles particle ]
particles-own
[
  vx vy vz                   ;; velocities rel axes
  speed mass energy          ;; particle info
  collision-time             ;; to determine when collision is
  collision-with             ;; to determine who the collision is with
  last-collision             ;; so they don't collide with one another many times
]

to setup
  clear-all
  ;; the wall shape is a custom 3D shape contained in "wall.txt"
  load-shapes-3d "wall.txt"
  set-default-shape particles "circle"
  set-default-shape walls "wall"
  set tick-delta .01
  set min-tick-delta .0000001
  make-box
  make-particles
  update-variables
  set init-avg-speed avg-speed
  set init-avg-energy avg-energy
  set open? false
  set avg-speed-cyan mean [speed] of turtles with [color = cyan]
  set avg-speed-magenta mean [speed] of turtles with [color = magenta]
  set avg-energy-cyan mean [energy] of turtles with [color = cyan]
  set avg-energy-magenta mean [ energy ] of turtles with [color = magenta]
  do-plotting
end

to go
  set collision-times [] ;; empty this out for new input
  ask particles
  [
    set collision-time tick-delta
    set collision-with nobody
      if collide? [
        detect-collisions
        detect-wall-collisions
      ]
  ]
  set collision-times sort collision-times

  ifelse first collision-times < tick-delta   ;; if something will collide before the tick
  [
    ask particles [ jump speed * first collision-times ] ; most particles to first collision
    tick-advance first collision-times ;; now, collide all the particles that are ready
    ask particles with [ collision-time = first collision-times ]
    [
      ifelse is-particle? collision-with
      [
        if collision-with > self [ ;; so that we don't collide the same particles twice
          collide collision-with
          set last-collision collision-with
          ask collision-with [ set last-collision myself ]
        ]
      ]
      [ wall-collide collision-with ]
    ]
  ]
  [
    ask particles [ jump speed * tick-delta ]
    tick-advance tick-delta
  ]

  ask particles [
    if last-collision != nobody and is-particle? last-collision
    [
      if distance last-collision > ( ( ( [size] of last-collision ) / 2 ) + ( size / 2 ) ) * 1.1
        [ set last-collision nobody ]
    ]
  ]

  if floor ticks > floor (ticks - tick-delta)
    [ update-variables ]

  set avg-speed-cyan mean [speed] of turtles with [color = cyan]
  set avg-speed-magenta mean [speed] of turtles with [color = magenta]
  set avg-energy-cyan mean [energy] of turtles with [color = cyan]
  set avg-energy-magenta mean [energy] of turtles with [color = magenta]

  do-plotting
  display
end

to update-variables
  set avg-speed  mean [speed] of particles
  set avg-energy  mean [energy] of particles
end

;;;
;;; distance and collision procedures
;;;

to detect-collisions ;; particle procedure

;; detect-collisions is a particle procedure that determines the time it takes to the collision between
;; two particles (if one exists).  It solves for the time by representing the equations of motion for
;; distance, velocity, and time in a quadratic equation of the vector components of the relative velocities
;; and changes in position between the two particles and solves for the time until the next collision

  let my-x-speed x-velocity heading pitch speed
  let my-y-speed y-velocity heading pitch speed
  let my-z-speed z-velocity pitch speed

  ask other particles with [self != [last-collision] of myself]
  [
    let dpx 0
    let dpy 0
    let dpz 0

   ;; since our world is wrapped, we can't just use calcs like xcor - my-x. Instead, we take the smallest
   ;; of either the wrapped or unwrapped distance for each dimension

    set dpx xcor - [xcor] of myself
    set dpy ycor - [ycor] of myself
    set dpz zcor - [zcor] of myself

    let x-speed x-velocity heading pitch speed  ;; speed of other particle in the x direction
    let y-speed y-velocity heading pitch speed  ;; speed of other particle in the y direction
    let z-speed z-velocity pitch speed          ;; speed of other particle in the z direction

    let dvx x-speed - my-x-speed ;; relative speed difference between particles in the x direction
    let dvy y-speed - my-y-speed ;; relative speed difference between particles in the y direction
    let dvz z-speed - my-z-speed ;; relative speed difference between particles in the z direction

    let sum-r ([size] of myself / 2) + (size / 2) ;; sum of both particle radii

    let p-squared   ((dpx * dpx) + (dpy * dpy) + (dpz * dpz)) - (sum-r ^ 2)   ;; p-squared represents difference of the
                                                                              ;; square of the radii and the square
                                                                              ;; of the initial positions

    let pv 2 * ((dpx * dvx) + (dpy * dvy) + (dpz * dvz))  ;;the vector product of the position times the velocity
    let v-squared (dvx * dvx) + (dvy * dvy) + (dvz * dvz) ;; the square of the difference in speeds
                                                          ;; represented as the sum of the squares of the x-component
                                                          ;; and y-component of relative speeds between the two particles

    ;; p-squared, pv, and v-squared are coefficients in the quadratic equation shown above that
    ;; represents how distance between the particles and relative velocity are related to the time,
    ;; t, at which they will next collide (or when their edges will just be touching)

    let D1 pv ^ 2 -  (4 * v-squared * p-squared)

    let time-to-collision  -1

    if D1 >= 0
      [ set time-to-collision (- pv - sqrt D1) / (2 * v-squared) ]

    if time-to-collision < tick-delta and time-to-collision > min-tick-delta
    [
      set collision-with myself
      set collision-time time-to-collision
      set collision-times lput time-to-collision collision-times
    ]
    if time-to-collision < min-tick-delta and time-to-collision > 0
    [
      set collision-with myself
      set collision-time min-tick-delta
      set collision-times lput min-tick-delta collision-times
    ]
  ]
end

to detect-wall-collisions ;; particle procedure
  update-component-vectors
  let my-vx vx * tick-delta
  let my-vy vy * tick-delta
  let my-vz vz * tick-delta

  detect-wall-collision "xy"
                        ( wall-max-pzcor - ( size / 2 ) )
                        ( wall-min-pzcor + ( size / 2 ) )
                        zcor
                        my-vz
  detect-wall-collision "yz"
                        ( wall-max-pxcor - ( size / 2 ) )
                        ( wall-min-pxcor + ( size / 2 ) )
                        xcor
                        my-vx
  detect-wall-collision "xz"
                        ( wall-max-pycor - ( size / 2 ) )
                        ( wall-min-pycor + ( size / 2 ) )
                        ycor
                        my-vy
  detect-divider-collision

  if collision-time < min-tick-delta [
    set collision-time min-tick-delta
  ]
  set collision-times lput collision-time collision-times
end

;; detect-wall-collision plane of wall, wall cor, wall cor, cor of particle, speed of particle
to detect-wall-collision [ plane max-wall min-wall cor cor-speed ] ;; particle procedure
  if ( cor + cor-speed ) > max-wall or
     ( cor + cor-speed ) < min-wall
  [
    let distance-to-wall abs( max-wall - cor )
    let time-fraction ( distance-to-wall / cor-speed )
    if ( time-fraction * tick-delta ) < collision-time and last-collision != plane
    [
      set collision-time time-fraction * tick-delta
      set collision-with plane
    ]
  ]
end

to detect-divider-collision ;; particle procedure
  let my-vx vx * tick-delta
  if xcor > 0 and my-vx < 0 and ( xcor + my-vx ) < (size / 2)
  [
    let distance-to-wall xcor - (size / 2)
    let time-fraction distance-to-wall / my-vx
    if ( time-fraction * tick-delta ) < collision-time and last-collision != "divider"
    [
      ;; where particle will be once it hits divider
      let future-ycor ycor + ( vy * time-fraction * tick-delta )
      let future-zcor zcor + ( vz * time-fraction * tick-delta )
      if not ( ( abs( future-ycor ) <= ( wall-max-pycor * opening-size / 100 ) and
                 abs( future-zcor ) <= ( wall-max-pzcor * opening-size / 100 ) ) and
               open? )
      [
        set collision-time time-fraction * tick-delta
        set collision-with "divider"
      ]
    ]
  ]
  if xcor < 0 and my-vx > 0 and ( xcor + my-vx ) > (- size / 2)
  [
    let distance-to-wall abs( xcor + (size / 2) )
    let time-fraction distance-to-wall / my-vx
    if ( time-fraction * tick-delta ) < collision-time and last-collision != "divider"
    [
      let future-ycor ycor + ( vy * time-fraction * tick-delta ) ;; where particle will be once it hits divider
      let future-zcor zcor + ( vz * time-fraction * tick-delta )
      if not ( ( abs( future-ycor ) <= ( wall-max-pycor * opening-size / 100 ) and
                 abs( future-zcor ) <= ( wall-max-pzcor * opening-size / 100 ) ) and
               open? )
      [
        set collision-time time-fraction * tick-delta
        set collision-with "divider"
      ]
    ]
  ]
end

to collide [ particle2 ] ;; particle procedure
  update-component-vectors
  ask particle2 [ update-component-vectors ]

  ;; find heading and pitch from the center of particle1 to the center of particle2
  let theading towards particle2
  let tpitch towards-pitch particle2

  ;; use these to determine the x, y, z components of theta
  let tx x-velocity theading tpitch 1
  let ty y-velocity theading tpitch 1
  let tz z-velocity tpitch 1

  ;; find the speed of particle1 in the direction of n
  let particle1-to-theta orth-projection vx vy vz tx ty tz

  ;; express particle1's movement along theta in terms of xyz
  let x1-to-theta particle1-to-theta * tx
  let y1-to-theta particle1-to-theta * ty
  let z1-to-theta particle1-to-theta * tz

  ;; now we can find the x, y and z components of the particle's velocity that
  ;; aren't in the direction of theta by subtracting the x, y, and z
  ;; components of the velocity in the direction of theta from the components
  ;; of the overall velocity of the particle
  let x1-opp-theta vx - x1-to-theta
  let y1-opp-theta vy - y1-to-theta
  let z1-opp-theta vz - z1-to-theta

  ;; do the same for particle2
  let particle2-to-theta orth-projection [vx] of particle2 [vy] of particle2 [vz] of particle2 tx ty tz

  let x2-to-theta particle2-to-theta * tx
  let y2-to-theta particle2-to-theta * ty
  let z2-to-theta particle2-to-theta * tz

  let x2-opp-theta [vx] of particle2 - x2-to-theta
  let y2-opp-theta [vy] of particle2 - y2-to-theta
  let z2-opp-theta [vz] of particle2 - z2-to-theta

  ;; calculate the velocity of the center of mass along theta
  let vcm ( ( mass * particle1-to-theta ) + ( [mass] of particle2 * particle2-to-theta ) )
            / ( mass + [mass] of particle2 )

  ;; switch momentums along theta
  set particle1-to-theta 2 * vcm - particle1-to-theta
  set particle2-to-theta 2 * vcm - particle2-to-theta

  ;; determine the x, y, z components of each particle's new velocities
  ;; in the direction of theta
  set x1-to-theta particle1-to-theta * tx
  set y1-to-theta particle1-to-theta * ty
  set z1-to-theta particle1-to-theta * tz

  set x2-to-theta particle2-to-theta * tx
  set y2-to-theta particle2-to-theta * ty
  set z2-to-theta particle2-to-theta * tz

  ;; now, we add the new velocities along theta to the unchanged velocities
  ;; opposite theta to determine the new heading, pitch, and speed of each particle
  set vx x1-to-theta + x1-opp-theta
  set vy y1-to-theta + y1-opp-theta
  set vz z1-to-theta + z1-opp-theta
  set heading vheading vx vy vz
  set pitch vpitch vx vy vz
  set speed vspeed vx vy vz
  set energy 0.5 * mass * speed ^ 2

  ask particle2 [
    set vx x2-to-theta + x2-opp-theta
    set vy y2-to-theta + y2-opp-theta
    set vz z2-to-theta + z2-opp-theta
    set heading vheading vx vy vz
    set pitch vpitch vx vy vz
    set speed vspeed vx vy vz
    set energy 0.5 * mass * speed ^ 2
  ]
end

to open-middle
  if opening-size = 20 [
    ask dividers [ set shape "opening20" ]
  ]
  if opening-size = 40 [
    ask dividers [ set shape "opening40" ]
  ]
  if opening-size = 60 [
    ask dividers [ set shape "opening60" ]
  ]
  if opening-size = 80 [
    ask dividers [ set shape "opening80" ]
  ]
  if opening-size = 100 [
    ask dividers [ die ]
  ]
  set open? true
end

to close-middle
  ask dividers [ set shape "flash" ]
  set open? false
end

to wall-collide [ collision-wall ] ;; particle procedure
  update-component-vectors

  ifelse collision-wall = "yz"
  [
    set heading vheading (- vx ) vy vz
  ][
  ifelse collision-wall = "xz"
  [
    set heading vheading vx (- vy ) vz
  ][
  ifelse collision-wall = "xy"
  [
    set pitch vpitch vx vy ( - vz )
   ]
   [
     set heading vheading (- vx ) vy vz
   ] ] ]
end

;;;
;;; drawing procedures
;;;

;; creates box
to make-box
  create-walls 1 [ ;; bottom wall
    set heading 0
    set zcor wall-min-pzcor
  ]
  create-walls 1 [ ;; top wall
    set heading 0
    set pitch 180
    set zcor wall-max-pzcor
  ]
  create-walls 1 [ ;; upper wall
    set heading 0
    set pitch 90
    set ycor wall-max-pycor
  ]
  create-walls 1 [ ;; lower wall
    set heading 0
    set pitch -90
    set ycor wall-min-pycor
  ]
  create-walls 1 [ ;; right wall
    set heading 0
    set roll -90
    set xcor wall-max-pxcor
  ]
  create-walls 1 [ ;; left wall
    set heading 0
    set roll 90
    set xcor wall-min-pxcor
  ]
  create-dividers 1 [ ;; center wall
    set heading 0
    set color grey
    set roll 90
    set xcor 0
    set shape "flash"
    set size box-width
  ]
  ask walls [
    set heading 0
    set color grey + random 3
    set size box-width
    set shape "wall"
  ]
end

;; creates initial particles
to make-particles
  create-particles num-magentas
  [
    setup-particle magenta-init-speed magenta-mass magenta
    random-position "left-half"
  ]
  create-particles num-cyans
  [
    setup-particle cyan-init-speed cyan-mass cyan
    random-position "right-half"
  ]
  check-initial-positions 0
  check-center-divider
end

to setup-particle [ my-speed my-mass my-color ] ;; particle procedure
  set speed my-speed
  set mass my-mass
  set energy 0.5 * mass * (speed ^ 2)
  set color my-color
  set size mass ^ 0.33
end

;; makes sure particles aren't overlapped at setup
to check-initial-positions [iterations]
  let check-again? false
  ask particles
  [
    if particle-overlap?
    [
      ifelse color = cyan
      [ random-position "right-half" ]
      [ random-position "left-half" ]
      set check-again? true
    ]
  ]
  ifelse iterations < 50
  [
    if check-again?
      [ check-initial-positions iterations + 1 ]
  ]
  [
    beep
    user-message "Not enough room for all these particles!"
  ]
end

;; makes sure particles don't go through center wall at setup
to check-center-divider
  let check-again? false
  ask particles
  [
    if abs( xcor ) < ( size / 2 )
    [
      ifelse color = cyan
      [ random-position "right-half" ]
      [ random-position "left-half" ]
      set check-again? true
    ]
  ]
  if check-again?
  [
    check-initial-positions 0
    check-center-divider
  ]
end
to-report particle-overlap? ;; particle procedure
  report any? other particles with [ distance myself <= ((size + [size] of myself) / 2 ) ]
end

;; place particle at random location inside the box.
to random-position [ side ] ;; particle procedure
  ifelse side = "left-half" [
    setxyz ( random-float ( ( box-width / 2 ) - 2 ) +
             ( wall-min-pxcor + 1 ) )
           ( random-float ( box-height - 2 ) +
             ( wall-min-pycor + 1 ) )
           ( random-float ( box-depth - 2 ) +
             ( wall-min-pzcor + 1 ) )
  ] [
    setxyz ( random-float ( ( box-width / 2 ) - 2 ) +
             ( 1 ) )
           ( random-float ( box-height - 2 ) +
             ( wall-min-pycor + 1 ) )
           ( random-float ( box-depth - 2 ) +
             ( wall-min-pzcor + 1 ) )
  ]
  tilt-up asin (1.0 - random-float 2.0)
  roll-right random-float 360
  update-component-vectors
  set heading vheading vx vy vz
  set pitch vpitch vx vy vz
end

;;;
;;; math procedures
;;;

;; consider the desired box-size
to-report box-width
  report ( world-width - 1 ) * ( box-size / 100 )
end

to-report box-height
  report ( world-height - 1 ) * ( box-size / 100 )
end

to-report box-depth
  report ( world-depth - 1 ) * ( box-size / 100 )
end

to-report wall-max-pxcor
  report max-pxcor * ( box-size / 100 )
end

to-report wall-max-pycor
  report max-pycor * ( box-size / 100 )
end

to-report wall-max-pzcor
  report max-pzcor * ( box-size / 100 )
end

to-report wall-min-pxcor
  report min-pxcor * ( box-size / 100 )
end

to-report wall-min-pycor
  report min-pycor * ( box-size / 100 )
end

to-report wall-min-pzcor
  report min-pzcor * ( box-size / 100 )
end

;; makes sure that the values stored in vx, vy, vz actually reflect
;; the appropriate heading, pitch, speed
to update-component-vectors ;; particle procedure
  set vx x-velocity heading pitch speed
  set vy y-velocity heading pitch speed
  set vz z-velocity pitch speed
end

;; reports velocity of a vector at a given angle and pitch
;; in the direction of x.
to-report x-velocity [ vector-angle vector-pitch vector-speed ]
  report sin( vector-angle ) * abs( cos( vector-pitch ) ) * vector-speed
end

;; reports velocity of a vector at a given angle and pitch
;; in the direction of y.
to-report y-velocity [ vector-angle vector-pitch vector-speed ]
  report cos( vector-angle ) * abs( cos( vector-pitch ) ) * vector-speed
end

;; reports velocity of a vector at a given angle and pitch
;; in the direction of z.
to-report z-velocity [ vector-pitch vector-speed ]
  report sin( vector-pitch ) * vector-speed
end

;; reports speed of a vector given xyz coords
to-report vspeed [ x y z ]
  report sqrt( x ^ 2 + y ^ 2 + z ^ 2 )
end

;; reports xt heading of a vector given xyz coords
to-report vheading [ x y z ]
  report atan x y
end

;; reports pitch of a vector given xyz coords
to-report vpitch [ x y z ]
  report asin ( z / ( vspeed x y z ) )
end

;; called by orthprojection
to-report dot-product [ x1 y1 z1 x2 y2 z2 ]
  report ( x1 * x2 ) + ( y1 * y2 ) + ( z1 * z2 )
end

;; component of 1 in the direction of 2 (Note order)
to-report orth-projection [ x1 y1 z1 x2 y2 z2 ]
  let dproduct dot-product x1 y1 z1 x2 y2 z2
  let speed-of-2 vspeed x2 y2 z2
  ;; if speed is 0 then there's no projection anyway
  ifelse speed-of-2 > 0
  [ report dproduct / speed-of-2 ]
  [ report 0 ]
end

;;;
;;; plotting procedures
;;;


to do-plotting
  set-current-plot "Average Speeds"
  set-current-plot-pen "cyan"
  plotxy ticks avg-speed-cyan
  set-current-plot-pen "magenta"
  plotxy ticks avg-speed-magenta

  set-current-plot "Average Energies"
  set-current-plot-pen "cyan"
  plotxy ticks avg-energy-cyan
  set-current-plot-pen "magenta"
  plotxy ticks avg-energy-magenta
end


; Copyright 2007 Uri Wilensky. All rights reserved.
; The full copyright notice is in the Information tab.
@#$#@#$#@
GRAPHICS-WINDOW
0
0
200
221
5
5
5.0
1
10
1
1
1
0
1
1
1
-5
5
-5
5
-5
5
1
1
1
ticks

BUTTON
40
222
133
255
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL

BUTTON
40
189
133
222
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

SLIDER
20
74
192
107
num-magentas
num-magentas
1
50
10
1
1
NIL
HORIZONTAL

PLOT
388
217
649
413
Average Energies
Time
Energy
0.0
1.0
0.0
500.0
true
true
PENS
"cyan" 10.0 0 -11221820 true
"magenta" 10.0 0 -5825686 true

PLOT
389
18
649
215
Average Speeds
Speed
Number
0.0
1.0
0.0
55.0
false
true
PENS
"cyan" 5.0 0 -11221820 true
"magenta" 5.0 0 -5825686 true

SLIDER
20
108
192
141
magenta-init-speed
magenta-init-speed
1
50
10
1
1
NIL
HORIZONTAL

SLIDER
20
142
192
175
magenta-mass
magenta-mass
1
10
1
1
1
NIL
HORIZONTAL

SLIDER
193
74
365
107
num-cyans
num-cyans
1
50
10
1
1
NIL
HORIZONTAL

SLIDER
193
108
365
141
cyan-init-speed
cyan-init-speed
0
50
46
1
1
NIL
HORIZONTAL

BUTTON
244
189
339
222
open
open-middle
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

BUTTON
244
222
339
255
close
close-middle
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

SWITCH
137
222
240
255
collide?
collide?
0
1
-1000

SLIDER
20
40
192
73
box-size
box-size
0
100
85
1
1
%
HORIZONTAL

SLIDER
193
40
365
73
opening-size
opening-size
0
100
80
20
1
%
HORIZONTAL

SLIDER
193
142
366
175
cyan-mass
cyan-mass
0
10
4
1
1
NIL
HORIZONTAL

MONITOR
12
259
181
304
magentas in left chamber
count particles with [ color = magenta and xcor < 0 ]
3
1
11

MONITOR
187
259
366
304
cyans in right chamber
count particles with [ color = cyan and xcor > 0 ]
3
1
11

MONITOR
187
312
366
357
average speed cyan
avg-speed-cyan
3
1
11

MONITOR
12
312
181
357
average speed magenta
avg-speed-magenta
3
1
11

MONITOR
187
365
366
410
average energy cyan
avg-energy-cyan
3
1
11

MONITOR
12
365
181
410
average energy magenta
avg-energy-magenta
3
1
11

@#$#@#$#@
WHAT IS IT?
-----------
This model is a 2D version of the 3D model Gas Lab Two Gas; it is one in a series of GasLab models.  They use the same basic rules for simulating the behavior of gases.  Each model integrates different features in order to highlight different aspects of gas behavior.

The basic principle of the models is that gas particles are assumed to have two elementary actions: they move and they collide - either with other particles or with any other objects such as walls.

This model is the simplest gas model in the suite of GasLab models.  The particles are moving and colliding with each other with no external constraints, such as gravity or containers.  In this model, particles are modeled as perfectly elastic ones with no energy except their kinetic energy -- which is due to their motion.  Collisions between particles are elastic.  Particles are colored according to their speed -- blue for slow, green for medium, and red for high.


HOW IT WORKS
------------
The basic principle of all GasLab models is the following algorithm (for more details, see the model "GasLab Gas in a Box"):

1) A particle moves in a straight line without changing its speed, unless it collides with another particle or bounces off the wall.
2) Two particles "collide" if their surfaces touch.  In this model, the time at which any collision is about to occur is measured, and particles move forward until the first pair to collide touch one another.  They are collided, and the cycle repeats.
3) The vector of collision for the particles describes the direction of the line connecting their centers.
4) The particles exchange momentum and energy only along this line, conforming to the conservation of momentum and energy for elastic collisions.
5) Each particle is assigned its new speed, direction and energy.


HOW TO USE IT
-------------
- OPEN: opens the door between the two chambers and allows particles to pas through
- CLOSE: closes the door separating the two chambers

- NUM-MAGENTAS and NUM-CYANS: the number of gas particles of each type.
- COLLIDE?: Turns collisions between particles on and off.
- MAGENTA-INIT-SPEED and CYAN-INIT-SPEED: the initial speed of each type of particle -- particles of the same type start with the same speed.
- MAGENTA-MASS and CYAN-MASS: the mass of each type particle -- particles of the same type have the same mass.
- BOX-SIZE: defines the size of the bounding box
- OPENING-SIZE: define the size of the "door" between the two chambers

As in most NetLogo models, the first step is to press SETUP. It puts in the initial conditions you have set with the sliders.  Be sure to wait till the SETUP button stops before pushing GO.
The GO button runs the models again and again.  This is a "forever" button.

Monitors:
MAGENTAS IN LEFT CHAMBER, CYANS IN RIGHT CHAMBER, AVERAGE SPEED MAGENTA and CYAN, and AVERAGE ENERGY MAGENTA and CYAN help you track the changes after the "door" has been opened.

Plots:
- Average Speeds: Shows the change in average speed for each type of particle.
- Average Energy: Shows the change in average energy for each type of particle.

Initially, all the particles have the same speed but random directions. Therefore the first histogram plots of speed and energy should show only one column each.  As the particles repeatedly collide, they exchange energy and head off in new directions, and the speeds are dispersed -- some particles get faster, some get slower, and the plot will show that change.


THINGS TO NOTICE
----------------
What variables affect how quickly the model reaches a new equilibrium when the wall is removed?

Why does the average speed for each color decrease as the model runs with the wall in place, even though the average energy remains constant?

What happens to the relative energies and speeds of each kind of particle as they intermingle?   What effect do the initial speeds and masses have on this relationship?

Does the system reach an equilibrium state?

Do heavier particles tend to have higher or lower speeds when the distribution of energy has reached equilibrium?

Is it reasonable to consider this box "insulated"?


THINGS TO TRY
-------------
Calculate how long the model takes to reach equilibrium with different sizes of windows (holding other parameters constant).

Calculate how long the model takes to reach equilibrium with different particle speeds.

Set the number of cyan particles to zero.  This is a model of a gas expanding into a vacuum.  This experiment was first done by Joule, using two insulated chambers separated by a valve.  He found that the temperature of the gas remained the same when the valve was opened.  Why would this be true? Is this model consistent with that observation?

Try some extreme situations, to test your intuitive understanding:
-- masses the same, speeds of the two particles very different.
-- speeds the same, masses very different.
-- a very small number of one kind of particle -- almost, but not quite a vacuum.  What happens to those few particles, and how do they affect the other kind?

Try relating quantitatively the ratio of the equilibrium speeds of both gases after the wall is opened to the ratio of the masses of both gases.  How are they related?


EXTENDING THE MODEL
-------------------
Monitor pressure in the right and left chambers.

Monitor temperature in the right and left chambers.

Replace the partition wall with a moveable piston, so that the two kinds of particles can push against each other without intermingling.  Do they arrive at a different equilibrium then?

Replace the partition wall with a surface that can transmit energy.

Add the histograms of energy and speed distribution (such as found in the "Free Gas" model).


NETLOGO FEATURES
-----------------
Notice the use of the histogram primitive.

When making 3D shapes, both sides of a shape must be defined or else one side becomes transparent. We use this feature to create a box with opaque inside walls and fencelike outside walls. For more information about 3d shapes, see the NetLogo User Manual.


CREDITS AND REFERENCES
----------------------
This was one of the original Connection Machine StarLogo applications (under the name GPCEE) and is now ported to NetLogo as part of the Participatory Simulations project.


HOW TO CITE
-----------
If you mention this model in an academic publication, we ask that you include these citations for the model itself and for the NetLogo software:
- Wilensky, U. (2007).  NetLogo GasLab Two Gas 3D model.  http://ccl.northwestern.edu/netlogo/models/GasLabTwoGas3D.  Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.
- Wilensky, U. (1999). NetLogo. http://ccl.northwestern.edu/netlogo/. Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.

In other publications, please use:
- Copyright 2007 Uri Wilensky. All rights reserved. See http://ccl.northwestern.edu/netlogo/models/GasLabTwoGas3D for terms of use.


COPYRIGHT NOTICE
----------------
Copyright 2007 Uri Wilensky. All rights reserved.

Permission to use, modify or redistribute this model is hereby granted, provided that both of the following requirements are followed:
a) this copyright notice is included.
b) this model will not be redistributed for profit without permission from Uri Wilensky. Contact Uri Wilensky for appropriate licenses for redistribution for profit.

This is a 3D version of the 2D model GasLab Two Gas.

@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 30 30 240

circle 2
false
0
Circle -7500403 true true 16 16 270
Circle -16777216 true false 46 46 210

clock
true
0
Circle -7500403 true true 30 30 240
Polygon -16777216 true false 150 31 128 75 143 75 143 150 158 150 158 75 173 75
Circle -16777216 true false 135 135 30

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

divider
false
0
Rectangle -7500403 true true 30 30 270 270

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flash
false
0
Rectangle -7500403 true true 30 30 270 270

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

link
true
0
Line -7500403 true 150 0 150 300

link direction
true
0
Line -7500403 true 150 150 30 225
Line -7500403 true 150 150 270 225

opening20
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 120 120 180 180

opening40
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 105 105 195 195

opening60
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 90 90 210 210

opening80
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 75 75 225 225

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 60 270 150 0 240 270 15 105 285 105
Polygon -7500403 true true 75 120 105 210 195 210 225 120 150 75

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wall
false
0
Rectangle -7500403 true true 30 30 270 270

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 3D 4.1pre7
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
