globals
[
  tick-delta          ;; how much we advance the tick counter this time through
  min-tick-delta      ;; the smallest tick-delta is allowed to be
  collision-times     ;; a list that of times of pending collisions

  ;; coordinates of the center of mass
  x-center
  y-center
  z-center
]

breed [ centers-of-mass center-of-mass ]

breed [ particles particle ]
particles-own
[
  ;; properties specific to particles
  speed
  energy
  mass

  collision-time ;; time in the future of the next collision
  collision-with ;; particle to collide with
  last-collision ;; particle we last collided with so we don't collide with the same particle many times

  ;; component vectors of the speed and direction of the particle
  x-vector
  y-vector
  z-vector
]

to setup
  clear-all
  create-particles 1 [
    set color pink
    set speed init-pink-speed
    setxyz 0 0 0
    set heading 0
    set pitch 0
    bk speed
    if wiggle? [ wiggle3d 5 ]
    set mass mass-of-pink
    set energy 0.5 * mass * speed ^ 2
    pen-down
    set last-collision nobody
  ]
  create-particles 1 [
    set color blue
    set speed init-blue-speed
    setxyz 0 0 0
    set heading collision-angle-xy
    set pitch collision-angle-z
    bk speed
    if wiggle? [ wiggle3d 5 ]
    set mass mass-of-blue
    set energy 0.5 * mass * speed ^ 2
    pen-down
    set last-collision nobody
  ]
  ask particles [
    set shape "circle"
    set size sqrt mass
  ]

  let total-mass sum [mass] of particles
  set x-center ( sum [ xcor * mass ] of particles ) / total-mass
  set y-center ( sum [ ycor * mass ] of particles ) / total-mass
  set z-center ( sum [ zcor * mass ] of particles ) / total-mass

  create-centers-of-mass 1 [
    set shape "x"
    set size 1
    set color gray
    setxyz x-center
           y-center
           z-center
    ifelse show-center-of-mass?
    [ pen-down ]
    [ hide-turtle ]
  ]

  set tick-delta .01
  set min-tick-delta 0
end

to go-mode
  if run-mode = "one-collision" [ go-once ]
  if run-mode = "all-collision-angles" [ go-all-angles ]
  if run-mode = "all-collision-pitches" [ go-all-angles-and-pitches ]
end

to go-once
  while [ go ]
    []
end

to go-all-angles
  let angle 15
  set wiggle? false
  repeat 23 [
    set collision-angle-xy angle
    setup
    while [ go ]
      []
    set angle angle + 15
  ]
end

to go-all-angles-and-pitches
  let zangle -90
  set wiggle? false
  repeat 11 [
    set collision-angle-z zangle
    setup
    while [ go ] [ ]
    set zangle zangle + 15
  ]
end

to-report go
  set collision-times [] ;; empty this out for new input
  ask particles
  [
    set collision-time tick-delta
    set collision-with nobody
    detect-collisions
  ]

  set x-center ( sum [ xcor * mass ] of particles ) / ( sum [mass] of particles )
  set y-center ( sum [ ycor * mass ] of particles ) / ( sum [mass] of particles )
  set z-center ( sum [ zcor * mass ] of particles ) / ( sum [mass] of particles )

  ask centers-of-mass
  [
    ifelse show-center-of-mass?  ;; marks a gray path along the particles' center of mass
      [ show-turtle
        pen-down ]
      [ hide-turtle
        pen-up ]
    setxyz x-center y-center z-center
  ]

  ifelse empty? collision-times
  [ set collision-times lput tick-delta collision-times ]
  [ set collision-times sort collision-times ]

  ifelse first collision-times < tick-delta   ;; if something will collide before the tick
  [
    tick-advance first collision-times
    ;; precheck that all the particles can move before we move any
    ;; this is so that the center of mass doesn't get slightly
    ;; off path because one particle moves and the other doesn't
    if any? particles with [ not can-move-cube? (speed * first collision-times) ]
      [ report false ]
    ask particles
    [
      jump speed * first collision-times
    ] ; most particles to first collision
    ask particle 0 [
      collide particle 1
      set last-collision collision-with
      ask particle 1 [ set last-collision myself ]
    ]
  ][
    ;; precheck that all the particles can move before we move any
    ;; this is so that the center of mass doesn't get slightly
    ;; off path because one particle moves and the other doesn't
    if any? particles with [ not can-move-cube? (speed * tick-delta) ]
      [ report false ]
    ask particles
    [
       jump speed * tick-delta
    ]
    tick-advance tick-delta
  ]

  ask particles [
    if last-collision != nobody
    [
      if distance last-collision > ( ([size] of last-collision / 2) + ( size / 2 ) ) * 1.5
      [ set last-collision nobody ]
    ]
  ]

  do-plotting
  display
  report true
end

;; we don't yet have different topologies in 3D so we have
;; to prevent the particles from wrapping around the world
;; manually (just like in pre-3.1 NetLogo).  We don't want the
;; particles to wrap around the world because it is confusing
;; when the center-of-mass is turned on.
to-report can-move-cube? [dist]
  let x xcor + (dx * dist)
  let y ycor + (dy * dist)
  let z zcor + (dz * dist)
  report not (x < min-pxcor + 0.5 or x > max-pxcor - 0.5 or
              y < min-pycor + 0.5 or y > max-pxcor - 0.5 or
              z < min-pzcor + 0.5 or z > max-pzcor - 0.5)
end

;;;
;;; distance and collision procedures
;;;

to detect-collisions ; particle procedure

;; detect-collisions is a particle procedure that determines the time it takes to the collision between
;; two particles (if one exists).  It solves for the time by representing the equations of motion for
;; distance, velocity, and time in a quadratic equation of the vector components of the relative velocities
;; and changes in position between the two particles and solves for the time until the next collision

  let my-x xcor
  let my-y ycor
  let my-z zcor
  let my-x-speed (x-velocity heading pitch speed )
  let my-y-speed (y-velocity heading pitch speed )
  let my-z-speed (z-velocity pitch speed )

  ask other particles with [self != [last-collision] of myself]
  [
    let dpx 0
    let dpy 0
    let dpz 0

   ;; since our world is wrapped, we can't just use calcs like xcor - my-x. Instead, we take the smallest
   ;; of either the wrapped or unwrapped distance for each dimension

    ifelse ( abs ( xcor - my-x ) <= abs ( ( xcor - my-x ) - world-width ) )
    [ set dpx (xcor - my-x) ]
    [ set dpx (xcor - my-x) - world-width ]  ;; relative distance between particles in the x direction
    ifelse ( abs ( ycor - my-y ) <= abs ( ( ycor - my-y ) - world-height ) )
    [ set dpy (ycor - my-y) ]
    [ set dpy (ycor - my-y) - world-height ]    ;; relative distance between particles in the y direction
    ifelse ( abs ( zcor - my-z ) <= abs ( ( zcor - my-z ) - world-depth ) )
    [ set dpz (zcor - my-z) ]
    [ set dpz (zcor - my-z) - world-depth ]       ;; relative distance between particles in the z direction

    let x-speed x-velocity heading pitch speed ;; speed of other particle in the x direction
    let y-speed y-velocity heading pitch speed ;; speed of other particle in the y direction
    let z-speed z-velocity pitch speed         ;; speed of other particle in the z direction

    let dvx x-speed - my-x-speed ;; relative speed difference between particles in the x direction
    let dvy y-speed - my-y-speed ;; relative speed difference between particles in the y direction
    let dvz z-speed - my-z-speed ;; relative speed difference between particles in the z direction

    let sum-r ([size] of myself / 2 ) + (size / 2) ;; sum of both particle radii

   ;; To figure out what the difference in position (P1) between two particles at a future time (t) would be,
   ;; one would need to know the current difference in position (P0) between the two particles
   ;; and the current difference in the velocity (V0) between of the two particles.

   ;; The equation that represents the relationship would be:   P1 = P0 + t * V0

   ;; we want find when in time (t), P1 would be equal to the sum of both the particle's radii (sum-r).
   ;; When P1 is equal to is equal to sum-r, the particles will just be touching each other at
   ;; their edges  (a single point of contact).

   ;; Therefore we are looking for when:   sum-r =  P0 + t * V0

   ;; This equation is not a simple linear equation, since P0 and V0 should both have x and y components
   ;;  in their two dimensional vector representation (calculated as dpx, dpy, and dvx, dvy).

   ;; By squaring both sides of the equation, we get:     (sum-r) * (sum-r) =  (P0 + t * V0) * (P0 + t * V0)

   ;;  When expanded gives:   (sum-r ^ 2) = (P0 ^ 2) + (t * PO * V0) + (t * PO * V0) + (t ^ 2 * VO ^ 2)

   ;;  Which can be simplified to:    0 = (P0 ^ 2) - (sum-r ^ 2) + (2 * PO * V0) * t + (VO ^ 2) * t ^ 2

   ;;  Below, we will let p-squared represent:   (P0 ^ 2) - (sum-r ^ 2)
   ;;  and pv represent: (2 * PO * V0)
   ;;  and v-squared represent: (VO ^ 2)

   ;;  then the equation will simplify to:     0 = p-squared + pv * t + v-squared * t^2

    let p-squared   ((dpx * dpx) + (dpy * dpy) + (dpz * dpz)) - (sum-r ^ 2)   ;; p-squared represents difference of the
                                                                              ;; square of the radii and the square
                                                                              ;; of the initial positions

    let pv  (2 * ((dpx * dvx) + (dpy * dvy) + (dpz * dvz)))  ;;the vector product of the position times the velocity
    let v-squared  ((dvx * dvx) + (dvy * dvy) + (dvz * dvz)) ;; the square of the difference in speeds
                                               ;; represented as the sum of the squares of the x-component
                                               ;; and y-component of relative speeds between the two particles

    ;; p-squared, pv, and v-squared are coefficients in the quadratic equation shown above that
    ;; represents how distance between the particles and relative velocity are related to the time,
    ;; t, at which they will next collide (or when their edges will just be touching)

    ;; Any quadratic equation that is the function of time (t), can represented in a general form as:
    ;;   a*t*t + b*t + c = 0,
    ;; where a, b, and c are the coefficients of the three different terms, and has solutions for t
    ;; that can be found by using the quadratic formula.  The quadratic formula states that if a is not 0,
    ;; then there are two solutions for t, either real or complex.

    ;; t is equal to (b +/- sqrt (b^2 - 4*a*c)) / 2*a

    ;; the portion of this equation that is under a square root is referred to here
    ;; as the determinant, D1.   D1 is equal to (b^2 - 4*a*c)
    ;; and:   a = v-squared, b = pv, and c = p-squared.

    let D1 pv ^ 2 -  (4 * v-squared * p-squared)

    ;; the next line next line tells us that a collision will happen in the future if
    ;; the determinant, D1 is >= 0,  since a positive determinant tells us that there is a
    ;; real solution for the quadratic equation.  Quadratic equations can have solutions
    ;; that are not real (they are square roots of negative numbers).  These are referred
    ;; to as imaginary numbers and for many real world systems that the equations represent
    ;; are not real world states the system can actually end up in.

    ;; Once we determine that a real solution exists, we want to take only one of the two
    ;; possible solutions to the quadratic equation, namely the smaller of the two the solutions:

    ;;  (b - sqrt (b^2 - 4*a*c)) / 2*a
    ;;  which is a solution that represents when the particles first touching on their edges.

    ;;  instead of (b + sqrt (b^2 - 4*a*c)) / 2*a
    ;;  which is a solution that represents a time after the particles have penetrated
    ;;  and are coming back out of each other and when they are just touching on their edges.


    let time-to-collision  -1

    if D1 >= 0
      [ set time-to-collision (- pv - sqrt D1) / (2 * v-squared) ]        ;;solution for time step

    ;; if time-to-collision is still -1 there is no collision in the future - no valid solution
    ;; note:  negative values for time-to-collision represent where particles would collide
    ;; if allowed to move backward in time.
    ;; if time-to-collision is greater than 1, then we continue to advance the motion
    ;; of the particles along their current trajectories.  They do not collide yet.
    ;; to keep the model from slowing down too much, if the particles are going to collide
    ;; at a time before min-tick-delta, just collide them a min-tick-delta instead

    if time-to-collision < tick-delta and time-to-collision > 0 [
      set collision-with myself
      set collision-time time-to-collision
      set collision-times lput time-to-collision collision-times
    ]
  ]
end

to collide [ particle2 ] ;; turtle procedure
  update-component-vectors
  ask particle2 [ update-component-vectors ]

  ;; find heading and pitch from the center of particle1 to the center of particle2
  let theading towards particle2
  let tpitch towards-pitch particle2

  ;; use these to determine the x, y, z components of theta
  let tx x-velocity theading tpitch 1
  let ty y-velocity theading tpitch 1
  let tz z-velocity tpitch 1

  ;; find the speed of particle1 in the direction of n
  let particle1-to-theta orth-projection x-vector y-vector z-vector tx ty tz

  ;; express particle1's movement along theta in terms of xyz
  let x1-to-theta particle1-to-theta * tx
  let y1-to-theta particle1-to-theta * ty
  let z1-to-theta particle1-to-theta * tz

  ;; now we can find the x, y and z components of the particle's velocity that
  ;; aren't in the direction of theta by subtracting the x, y, and z
  ;; components of the velocity in the direction of theta from the components
  ;; of the overall velocity of the particle
  let x1-opp-theta ( ( x-vector ) - ( x1-to-theta ) )
  let y1-opp-theta ( ( y-vector ) - ( y1-to-theta ) )
  let z1-opp-theta ( ( z-vector ) - ( z1-to-theta ) )

  ;; do the same for particle2
  let particle2-to-theta orth-projection [x-vector] of particle2 [y-vector] of particle2 [z-vector] of particle2 tx ty tz

  let x2-to-theta particle2-to-theta * tx
  let y2-to-theta particle2-to-theta * ty
  let z2-to-theta particle2-to-theta * tz

  let x2-opp-theta ( ( [x-vector] of particle2 ) - ( x2-to-theta ) )
  let y2-opp-theta ( ( [y-vector] of particle2 ) - ( y2-to-theta ) )
  let z2-opp-theta ( ( [z-vector] of particle2 ) - ( z2-to-theta ) )

  ;; calculate the velocity of the center of mass along theta
  let vcm ( ( ( mass * particle1-to-theta ) + ( [mass] of particle2 * particle2-to-theta ) )
            / ( mass + [mass] of particle2 ) )

  ;; switch momentums along theta
  set particle1-to-theta (2 * vcm - particle1-to-theta)
  set particle2-to-theta (2 * vcm - particle2-to-theta)

  ;; determine the x, y, z components of each particle's new velocities
  ;; in the direction of theta
  set x1-to-theta particle1-to-theta * tx
  set y1-to-theta particle1-to-theta * ty
  set z1-to-theta particle1-to-theta * tz

  set x2-to-theta particle2-to-theta * tx
  set y2-to-theta particle2-to-theta * ty
  set z2-to-theta particle2-to-theta * tz

  ;; now, we add the new velocities along theta to the unchanged velocities
  ;; opposite theta to determine the new heading, pitch, and speed of each particle
  set x-vector x1-to-theta + x1-opp-theta
  set y-vector y1-to-theta + y1-opp-theta
  set z-vector z1-to-theta + z1-opp-theta
  set heading vheading x-vector y-vector z-vector
  set pitch vpitch x-vector y-vector z-vector
  set speed vspeed x-vector y-vector z-vector
  set energy ( 0.5 * mass * speed ^ 2 )
  set shape "circle 2"
  stamp
  set shape "circle"

  ask particle2 [
    set x-vector x2-to-theta + x2-opp-theta
    set y-vector y2-to-theta + y2-opp-theta
    set z-vector z2-to-theta + z2-opp-theta
    set heading vheading x-vector y-vector z-vector
    set pitch vpitch x-vector y-vector z-vector
    set speed vspeed x-vector y-vector z-vector
    set energy ( 0.5 * mass * speed ^ 2 )
    set shape "circle 2"
    stamp
    set shape "circle"
  ]
end

;;;
;;; math procedures
;;;

;; makes sure that the values stored in vx, vy, vz actually reflect
;; the appropriate heading, pitch, speed
to update-component-vectors ;; particle procedure
  set x-vector speed * sin heading * cos pitch
  set y-vector speed * cos heading * cos pitch
  set z-vector speed * sin pitch
end

;; reports velocity of a vector at a given angle and pitch
;; in the direction of x.
to-report x-velocity [ vector-angle vector-pitch vector-speed ]
  report sin vector-angle * abs( cos vector-pitch ) * vector-speed
end

;; reports velocity of a vector at a given angle and pitch
;; in the direction of y.
to-report y-velocity [ vector-angle vector-pitch vector-speed ]
  report cos vector-angle * abs( cos vector-pitch ) * vector-speed
end

;; reports velocity of a vector at a given angle and pitch
;; in the direction of z.
to-report z-velocity [ vector-pitch vector-speed ]
  report sin vector-pitch * vector-speed
end

;; reports speed of a vector given xyz coords
to-report vspeed [ x y z ]
  report sqrt( x ^ 2 + y ^ 2 + z ^ 2 )
end

;; reports xt heading of a vector given xyz coords
to-report vheading [ x y z ]
  report atan x y
end

;; reports pitch of a vector given xyz coords
to-report vpitch [ x y z ]
  report round asin ( z / ( vspeed x y z ) )
end

;; called by orthprojection
to-report dot-product [ x1 y1 z1 x2 y2 z2 ]
  report ( ( x1 * x2 ) + ( y1 * y2 ) + ( z1 * z2 ) )
end

;; component of 1 in the direction of 2 (Note order)
to-report orth-projection [ x1 y1 z1 x2 y2 z2 ]
  let dproduct dot-product x1 y1 z1 x2 y2 z2
  let speed-of-2 ( vspeed x2 y2 z2 )
  ;; if speed is 0 then there's no projection anyway
  ifelse speed-of-2 > 0
  [ report ( dproduct / speed-of-2 ) ]
  [ report 0 ]
end

;; wiggle up to angle in random direction
to wiggle3d [ angle ]
  roll-right random-float 360
  tilt-up acos (1 - random-float ( 1 - cos angle ) )
end

;;; plotting procedure
to do-plotting
  set-current-plot "Speeds"

  ifelse [speed] of turtle 0 = [speed] of turtle 1
  [
    set-current-plot-pen "both"
    plotxy ticks [speed] of turtle 0
  ]
  [
    set-current-plot-pen "pink"
    plotxy ticks [speed] of turtle 0
    set-current-plot-pen "blue"
    plotxy ticks [speed] of turtle 1
  ]
end


; Copyright 2007 Uri Wilensky. All rights reserved.
; The full copyright notice is in the Information tab.
@#$#@#$#@
GRAPHICS-WINDOW
0
0
335
356
12
12
13.0
1
10
1
1
1
0
1
1
1
-12
12
-12
12
-12
12
1
1
1
ticks

BUTTON
283
82
347
127
go
go-mode
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

BUTTON
23
82
94
127
setup
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL

SLIDER
23
213
183
246
mass-of-pink
mass-of-pink
0.1
10
10
0.1
1
NIL
HORIZONTAL

SLIDER
188
213
346
246
mass-of-blue
mass-of-blue
0.1
10
5.4
0.1
1
NIL
HORIZONTAL

SLIDER
23
252
182
285
init-pink-speed
init-pink-speed
1
10
10
1
1
NIL
HORIZONTAL

SLIDER
188
252
346
285
init-blue-speed
init-blue-speed
1
10
10
1
1
NIL
HORIZONTAL

MONITOR
23
291
182
336
energy of pink
[energy] of particle 0
3
1
11

MONITOR
188
291
346
336
energy of blue
[energy] of particle 1
3
1
11

MONITOR
23
344
182
389
speed of pink
[speed] of particle 0
3
1
11

MONITOR
188
344
346
389
speed of blue
[speed] of particle 1
3
1
11

MONITOR
80
399
182
444
average speed
([speed] of particle 0 + [speed] of particle 1) / 2
3
1
11

MONITOR
188
399
296
444
average energy
([energy] of particle 0 + [energy] of particle 1) / 2
3
1
11

PLOT
24
455
346
605
speeds
time
speed
0.0
0.0
0.0
0.0
true
true
PENS
"pink" 1.0 0 -2064490 true
"blue" 1.0 0 -13345367 true
"both" 1.0 0 -16777216 true

SWITCH
88
44
288
77
show-center-of-mass?
show-center-of-mass?
1
1
-1000

SLIDER
23
135
346
168
collision-angle-xy
collision-angle-xy
15
345
105
15
1
degrees
HORIZONTAL

CHOOSER
100
82
277
127
run-mode
run-mode
"one-collision" "all-collision-angles" "all-collision-pitches"
0

SLIDER
23
174
346
207
collision-angle-z
collision-angle-z
-90
90
30
15
1
degrees
HORIZONTAL

SWITCH
88
10
288
43
wiggle?
wiggle?
0
1
-1000

@#$#@#$#@
WHAT IS IT?
-----------
This model is a 3D version of the 2D model GasLab Single Collision; it is one in a series of GasLab models. They use the same basic rules for simulating the behavior of gases. Each model integrates different features in order to highlight different aspects of gas behavior.

This model is simplified to show the collision of only two particles, since this event is so hard to watch when there are many particles in the world: given the initial motions of two colliding particles, what can we learn about their final motions from the principles of conservation of momentum and energy?


HOW IT WORKS
------------
The particles are modeled as hard balls with no internal energy except that which is due to their motion.  Collisions between particles are elastic.  Particles are colored according to speed -- blue for slow, green for medium, and red for high speeds.

Coloring of the particles is with respect to one speed (10).  Particles with a speed less than 5 are blue, those that are more than 15 are red, while all in those in-between are green.

Particles behave according to the following rules:
1. A particle moves in a straight line without changing its speed, unless it collides with another particle or bounces off the wall.  The particles are aimed to hit each other at the origin.
2. Two particles "collide" if they find themselves on the same patch (the world is composed of a grid of small squares called patches).
3. A random axis is chosen, as if they are two balls that hit each other and this axis is the line connecting their centers.
4. They exchange momentum and energy along that axis, according to the conservation of momentum and energy.  This calculation is done in the center of mass system.
5. Each turtle is assigned its new velocity, energy, and heading.
6. If a turtle finds itself on or very close to a wall of the container, it "bounces" -- that is, reflects its direction and keeps its same speed.


HOW TO USE IT
-------------
Initial settings:
- COLLISION-ANGLE: Sets the angle that separates the pink and blue particles before the collision.
- REFLECTION-ANGLE: Sets the angle of the axis connecting the particles' centers when they collide with respect to the vertical axis. To calculate the outcome of the collision, the speeds of the two particles are projected onto this new axis and the new speeds and headings are computed. Other GasLab models use random values for "REFLECTION-ANGLE", but this model allows you to experiment with them one by one. This angle is called THETA in the code of the model.
- INIT-PINK-SPEED (or BLUE): Sets the initial speed of the pink (or blue) particle.
- PINK-MASS (or BLUE): Sets the mass of the pink (or blue) particle.

Other settings:
- SHOW-CENTER-OF-MASS?: If ON, the center of mass of the system will be shown in gray.
- WIGGLE?: If ON, the initial particles will be placed in a random location to start.

Buttons for running the model:
- SETUP
- RUN-MODE: Chooses between ONE COLLISION (just one run), ALL-COLLISION-ANGLES (loops through all the collision angles with 15-degrees steps) and ALL-REFLECTION-ANGLES (loops through all the reflection angles with 15-degrees steps).
- GO

Monitors:
- ENERGY OF PINK (or -BLUE): Shows the current energy of the pink (or blue) particle.
- SPEED OF PINK (or -BLUE): Shows the current speed of the pink (or blue) particle.
- AVERAGE SPEED: Shows the average of the speeds of the two particles.
- TOTAL ENERGY: Shows the sum of the energies of the two particles.

Plots:
- SPEEDS: speed of each of the particles over time.


THINGS TO NOTICE
----------------
With COLLISION-ANGLE = 180 (directly across from each other) and REFLECTION-ANGLE = 90, it looks as if the two particles miss each other.  What is happening?

With REFLECTION-ANGLE = 45 degrees, the particles go off at right angles. Why?  Draw a picture of what is happening at the moment of collision.

With REFLECTION-ANGLE = 0 degrees, the two particles reverse direction. Why?

What is the motion of the center of mass?  What would you expect it to be?


THINGS TO TRY
-------------
Set the reflection-angle to zero.  Draw a picture representing the two balls as they collide, with their two faces touching.  Make the line connecting their centers be the same as theta.  Draw vectors representing their motion.

While running the following situations note the paths of the two particles.  Can you make sense of what they do?  Is it what you expected?

Choose a COLLISION-ANGLE and a REFLECTION-ANGLE and choose ONE-COLLISION to see one particular collision.

Choose a COLLISION-ANGLE and choose ALL-REFLECTION-ANGLES to cycle through all of the angles of reflection.

Choose a REFLECTION-ANGLE and choose ALL-COLLISION-ANGLES to cycle through all of the angles of collision.

Have the masses of the two particles be different.

Have the initial speeds of the two particles be different.

Change the initial positions and headings of the two particles.  As a simple case, set one on the y-axis and the other on the x-axis, (COLLISION-ANGLE = 90) each one heading toward the origin.  The center of mass is no longer stationary. Note its path.  Is it what you would expect?

If the center of mass is not stationary, the two particles often have different speeds after they collide, even when they have identical initial speeds and masses!  Why does this happen?  How can this satisfy the conservation of both energy and momentum?

The fact that the velocities are not always the same after every kind of collision is essential to getting a distribution of velocities among identical particles after many collisions, which is what we observe with particles in a gas.

Does this seem like a reasonable model for colliding particles? When is it reasonably valid, and when is it decidedly NOT valid?

When two particles collide, should theta be picked randomly -- each theta has an equal probability -- or in some other way?  Would this change the eventual velocity distribution among many particles?

After you have gotten used to observing and understanding these simple collisions, go to the "Free Gas" or "Gas in a Box" model.  Especially watch the particle whose path is traced in gray.  Does it make sense?  Can you picture each collision?

Record the velocities of each particle after each collision. After you have several sets of velocities, look at the entire velocity distribution. What do you notice? Is it the Maxwell-Boltzmann distribution?


EXTENDING THE MODEL
-------------------


NETLOGO FEATURES
----------------
Since there are not yet different topologies in NetLogo 3D we have to make sure that the particles don't go over the edge of the world using bounds checking code like in pre-3.1 NetLogo.


RELATED MODELS
--------------


HOW TO CITE
-----------
If you mention this model in an academic publication, we ask that you include these citations for the model itself and for the NetLogo software:
- Wilensky, U. (2007).  NetLogo GasLab Single Collision 3D model.  http://ccl.northwestern.edu/netlogo/models/GasLabSingleCollision3D.  Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.
- Wilensky, U. (1999). NetLogo. http://ccl.northwestern.edu/netlogo/. Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.

In other publications, please use:
- Copyright 2007 Uri Wilensky. All rights reserved. See http://ccl.northwestern.edu/netlogo/models/GasLabSingleCollision3D for terms of use.


COPYRIGHT NOTICE
----------------
Copyright 2007 Uri Wilensky. All rights reserved.

Permission to use, modify or redistribute this model is hereby granted, provided that both of the following requirements are followed:
a) this copyright notice is included.
b) this model will not be redistributed for profit without permission from Uri Wilensky. Contact Uri Wilensky for appropriate licenses for redistribution for profit.

This is a 3D version of the 2D model GasLab Single Collision.

@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

clock
true
0
Circle -7500403 true true 30 30 240
Polygon -16777216 true false 150 31 128 75 143 75 143 150 158 150 158 75 173 75
Circle -16777216 true false 135 135 30

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

link
true
0
Line -7500403 true 150 0 150 300

link direction
true
0
Line -7500403 true 150 150 30 225
Line -7500403 true 150 150 270 225

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 3D 4.1pre7
@#$#@#$#@
setup
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
