;; Artificial Neural Network Sandbox
;;
;; by L. F. Bencivengo, Jr.  December 14, 2020

globals [OutputA OutputB  OutputC
         OutputI OutputII OutputIII
         Output1 Output2  Output3
         started? auto_Input_Pattern input#]

breed [Input_Neurons Input_Neuron]
breed [Hidden_Neurons Hidden_Neuron]
breed [Output_Neurons Output_Neuron]
turtles-own [threshold weight1 weight2 weight3 activation output]

directed-link-breed [axons-AI axon-AI]
directed-link-breed [axons-BII axon-BII]
directed-link-breed [axons-CIII axon-CIII]

;; BUTTONS: This section defines the function of each Interface Button
to setup
  clear-all
  ;; turn off auto_Input_Pattern
  set auto_Input_Pattern FALSE
  set input# 0
  ;; create neurons with current output = 0
  set-default-shape Input_Neurons "square 2"
  set-default-shape Hidden_Neurons "triangle 2"
  set-default-shape Output_Neurons "circle 2"
  setup-Input_Neurons
  setup-Hidden_Neurons
  setup-Output_Neurons
  ;; create axons that connect each neuron in the network
  set-default-shape axons-AI "axon-ai-0"
  set-default-shape axons-BII "axon-bii-0"
  set-default-shape axons-CIII "axon-ciii-0"
  setup-synapses
  ;; clear the ticks to disable Input Buttons
  set started? FALSE
  clear-ticks
end

to go
  ;; if just starting, reset ticks and enable Input Buttons
  if not started? [reset-ticks]
  set started? TRUE
  ;; if auto-input is clicked, run through the auto-input sequence
  ;; read values of weights and thresholds from sliders
  update-weights
  ;; update the activation and output of each neuron in the network
  update-Hidden_Outputs
  update-Output_Outputs
  ;; update the display to show any changes in neurons' output
  update-display
  tick
end

to Input_A
  ;; change the current state of Input Neuron A
  ifelse OutputA = 0 [set OutputA 1] [set OutputA 0]
  ifelse OutputA = 0 [ask turtle 0 [set output 0]] [ask turtle 0 [set output 1]]
  stop
end

to Input_B
  ;; change the current state of Input Neuron B
  ifelse OutputB = 0 [set OutputB 1] [set OutputB 0]
  ifelse OutputB = 0 [ask turtle 1 [set output 0]] [ask turtle 1 [set output 1]]
end

to Input_C
  ;; change the current state of Input Neuron C
  ifelse OutputC = 0 [set OutputC 1] [set OutputC 0]
  ifelse OutputC = 0 [ask turtle 2 [set output 0]] [ask turtle 2 [set output 1]]
end

to reset_weights
  ;; reset the sliders for every neuron's weights and thresholds to the default value of 1
  set weightIA 1 set weightIB 1 set weightIC 1
  set thresholdI 1
  set weightIIA 1 set weightIIB 1 set weightIIC 1
  set thresholdII 1
  set weightIIIA 1 set weightIIIB 1 set weightIIIC 1
  set thresholdIII 1
  set weight1I 1 set weight1II 1 set weight1III 1
  set threshold1 1
  set weight2I 1 set weight2II 1 set weight2III 1
  set threshold2 1
  set weight3I 1 set weight3II 1 set weight3III 1
  set threshold3 1
end

;; PROCEDURES: This section defines the Procedures which control the model

to setup-Input_Neurons
  ;; turtle 0 is Input Neuron A
  create-Input_Neurons 1 [
    set size 3  set color grey
    set xcor -8  set ycor 12]
  ;; turtle 1 is Input Neuron B
  create-Input_Neurons 1 [
    set size 3  set color grey
    set xcor 0  set ycor 12]
  ;; turtle 2 is Input Neuron C
  create-Input_Neurons 1 [
    set size 3  set color grey
    set xcor 8  set ycor 12]
end

to setup-Hidden_Neurons
  ;; turtle 3 is Hidden Neuron I
  create-Hidden_Neurons 1 [
    set size 3  set color grey
    set xcor -8  set ycor 0]
  ;; turtle 4 is Hidden Neuron II
  create-Hidden_Neurons 1 [
    set size 3  set color grey
    set xcor 0  set ycor 0]
  ;; turtle 5 is Hidden Neuron III
  create-Hidden_Neurons 1 [
    set size 3  set color grey
    set xcor 8  set ycor 0]
end

to setup-Output_Neurons
  ;; turtle 6 is Output Neuron 1
  create-Output_Neurons 1 [
    set size 3  set color grey
    set xcor -8  set ycor -12]
  ;; turtle 7 is Output Neuron 2
  create-Output_Neurons 1 [
    set size 3  set color grey
    set xcor 0  set ycor -12]
  ;; turtle 8 is Output Neuron 3
  create-Output_Neurons 1 [
    set size 3  set color grey
    set xcor 8  set ycor -12]
end

to setup-synapses
  ;; create axons from each Input Neuron to each Hidden Neuron
  ask turtle 0 [create-axons-AI-to Hidden_Neurons]
  ask turtle 1 [create-axons-BII-to Hidden_Neurons]
  ask turtle 2 [create-axons-CIII-to Hidden_Neurons];; [set thickness 0.25 set color white]]
  ;; create axons from each Hidden Neuron to each Output Neuron
  ask turtle 3 [create-axons-AI-to Output_Neurons]
  ask turtle 4 [create-axons-BII-to Output_Neurons]
  ask turtle 5 [create-axons-CIII-to Output_Neurons]
end

to update-Input_Pattern
  ;; when auto_Input_Pattern = TRUE, runs through a sequence of input patterns
  ;; changing the pattern once every second
  ifelse input# = 8 [set input# 1] [set input# input# + 1]
  (ifelse
    input# = 1 [ask turtle 0 [set output 0] ask turtle 1 [set output 0] ask turtle 2 [set output 0]
                set OutputA 0 set OutputB 0 set OutputC 0]
    input# = 2 [ask turtle 0 [set output 1] ask turtle 1 [set output 0] ask turtle 2 [set output 0]
                set OutputA 1 set OutputB 0 set OutputC 0]
    input# = 3 [ask turtle 0 [set output 0] ask turtle 1 [set output 1] ask turtle 2 [set output 0]
                set OutputA 0 set OutputB 1 set OutputC 0]
    input# = 4 [ask turtle 0 [set output 0] ask turtle 1 [set output 0] ask turtle 2 [set output 1]
                set OutputA 0 set OutputB 0 set OutputC 1]
    input# = 5 [ask turtle 0 [set output 1] ask turtle 1 [set output 1] ask turtle 2 [set output 0]
                set OutputA 1 set OutputB 1 set OutputC 0]
    input# = 6 [ask turtle 0 [set output 1] ask turtle 1 [set output 0] ask turtle 2 [set output 1]
                set OutputA 1 set OutputB 0 set OutputC 1]
    input# = 7 [ask turtle 0 [set output 0] ask turtle 1 [set output 1] ask turtle 2 [set output 1]
                set OutputA 0 set OutputB 1 set OutputC 1]
    input# = 8 [ask turtle 0 [set output 1] ask turtle 1 [set output 1] ask turtle 2 [set output 1]
                set OutputA 1 set OutputB 1 set OutputC 1]
               [])
end

to update-weights
  ;; read weights from slider values:
  ;; Ex. "weight1" is a neuron variable while "weightIA" is the value read from a slider
  ;; update Hidden Neuron weights
  ask turtle 3 [set weight1 weightIA set weight2 weightIB set weight3 weightIC set threshold thresholdI]
  ask turtle 4 [set weight1 weightIIA set weight2 weightIIB set weight3 weightIIC set threshold thresholdII]
  ask turtle 5 [set weight1 weightIIIA set weight2 weightIIIB set weight3 weightIIIC set threshold thresholdIII]
  ;; update Output Neuron weights
  ask turtle 6 [set weight1 weight1I set weight2 weight1II set weight3 weight1III set threshold threshold1]
  ask turtle 7 [set weight1 weight2I set weight2 weight2II set weight3 weight2III set threshold threshold2]
  ask turtle 8 [set weight1 weight3I set weight2 weight3II set weight3 weight3III set threshold threshold3]
end

to update-Hidden_Outputs
  ;; multiply the value of each input to a Hidden Neuron by the weight of its synapse
  ;; compare the weighted sum to the threshold to determine the neuron's output (0 or 1)
  ask turtle 3 [
    set activation (weight1 * OutputA + weight2 * OutputB + weight3 * OutputC)
    ifelse activation > threshold [set output 1] [set output 0]
    ifelse output = 1 [set OutputI 1] [set OutputI 0]
  ]
  ask turtle 4 [
    set activation (weight1 * OutputA + weight2 * OutputB + weight3 * OutputC)
    ifelse activation > threshold [set output 1] [set output 0]
    ifelse output = 1 [set OutputII 1] [set OutputII 0]
  ]
  ask turtle 5 [
    set activation (weight1 * OutputA + weight2 * OutputB + weight3 * OutputC)
    ifelse activation > threshold [set output 1] [set output 0]
    ifelse output = 1 [set OutputIII 1] [set OutputIII 0]
  ]
end

to update-Output_Outputs
  ;; multiply the value of each input to an Ouput Neuron by the weight of its synapse
  ;; compare the weighted sum to the threshold to determine the neuron's output (0 or 1)
  ask turtle 6 [
    set activation (weight1 * OutputI + weight2 * OutputII + weight3 * OutputIII)
    ifelse activation > threshold [set output 1] [set output 0]
    ifelse output = 1 [set Output1 1] [set Output1 0]
  ]
  ask turtle 7 [
    set activation (weight1 * OutputI + weight2 * OutputII + weight3 * OutputIII)
    ifelse activation > threshold [set output 1] [set output 0]
    ifelse output = 1 [set Output2 1] [set Output2 0]
  ]
  ask turtle 8 [
    set activation (weight1 * OutputI + weight2 * OutputII + weight3 * OutputIII)
    ifelse activation > threshold [set output 1] [set output 0]
    ifelse output = 1 [set Output2 1] [set Output2 0]
  ]
end

to update-display
  ;; changes each neuron's shape and color to display whether its output is 0 or 1
  ask Input_Neurons [
    ifelse output = 1
    [set shape "square" set color white ask my-out-links [set color white]]
    [set shape "square 2" set color grey ask my-out-links [set color grey]]
  ]
  ask Hidden_Neurons [
    ifelse output = 1
    [set shape "triangle" set color white ask my-out-links [set color white]]
    [set shape "triangle 2" set color grey ask my-out-links [set color grey]]
  ]
  ask Output_Neurons [
    ifelse output = 1
    [set shape "circle" set color white ask my-out-links [set color white]]
    [set shape "circle 2" set color grey ask my-out-links [set color grey]]
  ]
end
@#$#@#$#@
GRAPHICS-WINDOW
6
37
339
475
-1
-1
13.0
1
10
1
1
1
0
0
0
1
-12
12
-16
16
0
0
1
ticks
30.0

BUTTON
463
27
526
60
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
533
27
596
60
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
352
181
524
214
weightIA
weightIA
-50
50
1.0
1
1
NIL
HORIZONTAL

SLIDER
352
214
524
247
weightIB
weightIB
-50
50
1.0
1
1
NIL
HORIZONTAL

SLIDER
352
247
524
280
weightIC
weightIC
-50
50
1.0
1
1
NIL
HORIZONTAL

TEXTBOX
384
160
494
178
Hidden Neuron I
14
0.0
1

SLIDER
352
280
524
313
thresholdI
thresholdI
-10
10
1.0
1
1
NIL
HORIZONTAL

SLIDER
542
181
714
214
weightIIA
weightIIA
-50
50
1.0
1
1
NIL
HORIZONTAL

TEXTBOX
573
158
684
176
Hidden Neuron II
14
0.0
1

SLIDER
542
214
714
247
weightIIB
weightIIB
-50
50
1.0
1
1
NIL
HORIZONTAL

SLIDER
542
247
714
280
weightIIC
weightIIC
-50
50
1.0
1
1
NIL
HORIZONTAL

SLIDER
542
280
714
313
thresholdII
thresholdII
-10
10
1.0
1
1
NIL
HORIZONTAL

TEXTBOX
759
162
868
180
Hidden Neuron III
14
0.0
1

SLIDER
729
182
901
215
weightIIIA
weightIIIA
-50
50
1.0
1
1
NIL
HORIZONTAL

SLIDER
729
215
901
248
weightIIIB
weightIIIB
-50
50
1.0
1
1
NIL
HORIZONTAL

SLIDER
729
248
901
281
weightIIIC
weightIIIC
-50
50
1.0
1
1
NIL
HORIZONTAL

SLIDER
729
281
901
314
thresholdIII
thresholdIII
-10
10
1.0
1
1
NIL
HORIZONTAL

SLIDER
352
345
524
378
weight1I
weight1I
-50
50
1.0
1
1
NIL
HORIZONTAL

TEXTBOX
383
323
533
341
\t\tOutput Neuron 1
14
0.0
1

SLIDER
352
378
524
411
weight1II
weight1II
-50
50
1.0
1
1
NIL
HORIZONTAL

SLIDER
352
411
524
444
weight1III
weight1III
-50
50
1.0
1
1
NIL
HORIZONTAL

SLIDER
352
443
524
476
threshold1
threshold1
-10
10
1.0
1
1
NIL
HORIZONTAL

SLIDER
542
345
714
378
weight2I
weight2I
-50
50
1.0
1
1
NIL
HORIZONTAL

SLIDER
542
378
714
411
weight2II
weight2II
-50
50
1.0
1
1
NIL
HORIZONTAL

SLIDER
542
411
714
444
weight2III
weight2III
-50
50
1.0
1
1
NIL
HORIZONTAL

SLIDER
542
444
714
477
threshold2
threshold2
-10
10
1.0
1
1
NIL
HORIZONTAL

SLIDER
730
345
902
378
weight3I
weight3I
-50
50
1.0
1
1
NIL
HORIZONTAL

SLIDER
730
378
902
411
weight3II
weight3II
-50
50
1.0
1
1
NIL
HORIZONTAL

SLIDER
730
411
902
444
weight3III
weight3III
-50
50
1.0
1
1
NIL
HORIZONTAL

SLIDER
730
444
902
477
threshold3
threshold3
-10
10
1.0
1
1
NIL
HORIZONTAL

TEXTBOX
572
323
722
341
Output Neuron 2\t\t
14
0.0
1

TEXTBOX
760
324
910
342
Output Neuron 3
14
0.0
1

TEXTBOX
38
11
321
34
Artificial Neural Network Sandbox
16
0.0
1

BUTTON
395
85
472
118
NIL
Input_A
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

BUTTON
584
86
659
119
NIL
Input_B
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

BUTTON
771
86
848
119
NIL
Input_C
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

BUTTON
880
29
990
62
NIL
reset_weights
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
643
27
785
60
auto_Input_Pattern
every 1 [update-Input_Pattern]
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

@#$#@#$#@
## WHAT IS IT?

This model simulates a simple Artificial Neural Network (ANN) which we would normally build in class using Arduino microcontrollers to serve as the neurons.  The user can set the weights and thresholds of the artificial neurons, thus determining the behavior of the ANN.

## HOW IT WORKS

(what rules the agents use to create the overall behavior of the model)

## HOW TO USE IT

(how to use the model, including a description of each of the items in the Interface tab)

## THINGS TO NOTICE

(suggested things for the user to notice while running the model)

## THINGS TO TRY

(suggested things for the user to try to do (move sliders, switches, etc.) with the model)

## EXTENDING THE MODEL

(suggested things to add or change in the Code tab to make the model more complicated, detailed, accurate, etc.)

## NETLOGO FEATURES

(interesting or unusual features of NetLogo that the model uses, particularly in the Code tab; or where workarounds were needed for missing features)

## RELATED MODELS

(models in the NetLogo Models Library and elsewhere which are of related interest)

## CREDITS AND REFERENCES

(a reference to the model's URL on the web if it has one, as well as any other necessary credits, citations, and links)
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.1.1
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

axon-ai-0
0.0
-0.2 0 0.0 1.0
0.0 1 4.0 4.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

axon-ai-1
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

axon-bii-0
0.0
-0.2 1 4.0 4.0
0.0 0 0.0 1.0
0.2 1 4.0 4.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

axon-bii-1
0.0
-0.2 1 1.0 0.0
0.0 0 0.0 1.0
0.2 1 1.0 0.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

axon-ciii-0
0.0
-0.2 1 4.0 4.0
0.0 1 4.0 4.0
0.2 1 4.0 4.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

axon-ciii-1
0.0
-0.2 1 1.0 0.0
0.0 1 1.0 0.0
0.2 1 1.0 0.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
