;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; STUDENT MILITARY SERVICE MOTIVATION MODEL
;; Based on research on motivations of students in Kyiv universities
;; to join military service after Russia's full-scale invasion of Ukraine
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;; GLOBAL VARIABLES
globals [
  war-week              ; Current week count since full-scale invasion
  media-influence       ; Level of positive media coverage of military (0-100)
  economic-hardship     ; Level of economic difficulty in the country (0-100)

  ;; Regional distribution percentages
  percent-west          ; Percentage of students from Western Ukraine
  percent-central       ; Percentage of students from Central Ukraine
  percent-east          ; Percentage of students from Eastern Ukraine
  percent-south         ; Percentage of students from Southern Ukraine

  ;; Event counters for tracking simulation events
  military-events       ; Count of significant military events
  economic-events       ; Count of significant economic events
  political-events      ; Count of significant political events
]

;; AGENT DEFINITIONS
breed [students student]

students-own [
  ;; Demographic attributes
  age                   ; Student age (17-26)
  gender                ; Student gender ("male" or "female")
  study-year            ; Year of study (1-4)
  region                ; Home region ("West", "Central", "East", "South")

  ;; Background factors (0-100 scale)
  family-influence      ; Level of patriotic/military influence from family
  educational-influence ; Level of patriotic/educational influence from school/university
  activism-involvement  ; Level of involvement in civic/political organizations

  ;; Motivation types (0-100 scale)
  social-motivation     ; Motivation from social environment and peers
  moral-motivation      ; Moral and psychological motivation
  political-motivation  ; Civic and political motivation
  economic-motivation   ; Economic/financial motivation

  ;; Decision variables
  motivation-threshold  ; Individual threshold required to join (varies by person)
  joined?               ; Whether student has joined military service

  ;; Network variables
  my-organization       ; Type of organization the student belongs to (if any)
  my-university         ; University the student attends
  peers                 ; List of connected peers
  my-peers-joined       ; Percentage of peers who have joined military
]

patches-own [
  university            ; University identifier (name)
  is-organization?      ; Whether patch represents an organization
  organization-type     ; Type of organization if applicable ("youth", "political", "volunteer")
]

;; MODEL SETUP PROCEDURES
to setup
  clear-all

  ;; Initialize regional distribution
  set percent-west 30    ; 30% of students from Western region
  set percent-central 40 ; 40% from Central region
  set percent-east 20    ; 20% from Eastern region
  set percent-south 10   ; 10% from Southern region

  set economic-hardship 80 ; Start with high economic hardship (wartime conditions)

  ;; Initialize event counters
  set military-events 0
  set economic-events 0
  set political-events 0

  ;; Set up the model environment and agents
  setup-environment
  setup-students
  setup-organizations
  reset-ticks
end

;; Sets up the physical environment (universities and spaces)
to setup-environment
  ;; Initialize all patches to default state
  ask patches [
    set university "none"
    set is-organization? false
    set organization-type "none"
    set pcolor brown - 2  ; Default background color
  ]

  ;; Create university areas on the map
  setup-universities
end

;; Creates student agents with appropriate attributes
to setup-students
  create-students population-size [
    set shape "person"
    set size 1  ; Set visible size for students

    ;; Initialize student attributes (demographics, motivations, etc.)
    setup-student-attributes

    ;; Place student at an appropriate university
    let target-university one-of patches with [university != "none"]

    ;; Fallback if no universities exist
    if target-university = nobody [
      move-to one-of patches
    ]

    ;; Place student at their assigned university
    if target-university != nobody [
      set my-university [university] of target-university

      ;; Try to find an unoccupied patch in the university
      let target-patch one-of patches with [(university = [my-university] of myself) and not any? students-here]

      ;; If all university patches are occupied, try nearby patches
      if target-patch = nobody [
        set target-patch one-of patches with [university = [my-university] of myself]

        ;; Last resort fallback
        if target-patch = nobody [
          set target-patch one-of patches
        ]
      ]

      ;; Move to the selected patch
      move-to target-patch

      ;; Add small random movement to avoid perfect alignment
      fd random-float 2
    ]
  ]
end

;; Initialize individual student attributes based on research data
to setup-student-attributes
  ;; Set demographic attributes
  set age 17 + random 10  ; Ages 17-26 (typical university age range)
  set gender ifelse-value (random 100 < percent-female) ["female"] ["male"]
  set study-year 1 + random 4  ; Years 1-4 of university

  ;; Set regional background with appropriate probabilities
  let random-num random 100
  if random-num < percent-west [set region "West"]
  if random-num >= percent-west and random-num < (percent-west + percent-central) [set region "Central"]
  if random-num >= (percent-west + percent-central) and random-num < (percent-west + percent-central + percent-east) [set region "East"]
  if random-num >= (percent-west + percent-central + percent-east) [set region "South"]

  ;; Initialize influence factors based partly on region
  set family-influence setup-family-influence
  set educational-influence setup-educational-influence
  set activism-involvement 0 ; Will be updated when organizations form

  ;; Initialize motivation factors with gender differences based on research
  ifelse gender = "female" [
    ;; Female students - research shows higher moral components
    set social-motivation random 30 + 5
    set moral-motivation random 30 + 10
    set political-motivation random 30
    set economic-motivation random 30 - 5
  ] [
    ;; Male students
    set social-motivation random 30
    set moral-motivation random 30
    set political-motivation random 30 + 5
    set economic-motivation random 30
  ]

  ;; Set individual threshold (varies by person and gender)
  ifelse gender = "female" [
    ;; Women typically require higher motivation due to gender barriers in military
    set motivation-threshold 60 + random 40
  ] [
    ;; Men's threshold
    set motivation-threshold 50 + random 50
  ]

  ;; Initialize decision and network variables
  set joined? false
  set peers []
end

;; Calculate family influence based on regional background
to-report setup-family-influence
  let base-influence random 60

  ;; Regional effects on family influence based on research
  if region = "West" [set base-influence base-influence + 20]  ; Western regions more patriotic
  if region = "South" or region = "East" [set base-influence max list 0 (base-influence - 10)]  ; Less patriotic influence

  report min list 100 base-influence  ; Cap at 100
end

;; Calculate educational influence based on university quality and region
to-report setup-educational-influence
  let base-influence random 50
  let uni my-university

  ;; Higher quality education in prestigious universities
  if uni = "NaUKMA" [set base-influence base-influence + 20]  ; Kyiv-Mohyla Academy
  if uni = "KNU" [set base-influence base-influence + 15]     ; Kyiv National University
  if uni = "LNU" [set base-influence base-influence + 10]     ; Lviv National University

  ;; Regional effects on educational influence
  if region = "West" [set base-influence base-influence + 5]  ; Western universities more nationally oriented

  report min list 100 base-influence  ; Cap at 100
end

;; Create organizations (youth, political, volunteer) on the map
to setup-organizations
  ;; Number of each organization type
  let num-youth-orgs 3 + random 2      ; 3-4 youth organizations
  let num-political-orgs 2 + random 3  ; 2-4 political organizations
  let num-volunteer-orgs 4 + random 3  ; 4-6 volunteer organizations

  ;; Create youth organizations
  ask n-of num-youth-orgs patches with [not is-organization? and university != "none"] [
    set is-organization? true
    set organization-type "youth"
    set pcolor yellow
  ]

  ;; Create political organizations
  ask n-of num-political-orgs patches with [not is-organization? and university != "none"] [
    set is-organization? true
    set organization-type "political"
    set pcolor blue
  ]

  ;; Create volunteer organizations
  ask n-of num-volunteer-orgs patches with [not is-organization? and university != "none"] [
    set is-organization? true
    set organization-type "volunteer"
    set pcolor green
  ]

  ;; Students join organizations based on their attributes and proximity
  ask students [
    ;; Find nearby organizations (within radius of 5 patches)
    let nearby-orgs patches in-radius 5 with [is-organization?]

    ;; If there are nearby organizations, possibly join one based on student characteristics
    if any? nearby-orgs [
      let chosen-org one-of nearby-orgs
      let join-probability 0

      ;; Calculate join probability based on organization type and student characteristics
      if [organization-type] of chosen-org = "youth" [
        ;; Younger students more likely to join youth orgs
        set join-probability 70 - ((age - 17) * 10)
      ]

      if [organization-type] of chosen-org = "political" [
        ;; Political orgs appeal to students with political motivation
        set join-probability political-motivation / 2
      ]

      if [organization-type] of chosen-org = "volunteer" [
        ;; Volunteer orgs appeal more to students with moral motivation
        set join-probability moral-motivation / 2
      ]

      ;; Regional effects on organization joining
      if region = "West" and [organization-type] of chosen-org = "volunteer" [
        set join-probability join-probability + 20  ; Western regions more volunteer-oriented
      ]

      if region = "East" and [organization-type] of chosen-org = "political" [
        set join-probability join-probability + 15  ; Eastern regions more politically active
      ]

      ;; Join organization if probability threshold met
      if random 100 < join-probability [
        set my-organization [organization-type] of chosen-org

        ;; Being in an organization increases activism involvement
        set activism-involvement 20 + random 40

        ;; Different organization types affect different motivations
        if my-organization = "youth" [
          set social-motivation social-motivation + 10
        ]
        if my-organization = "political" [
          set political-motivation political-motivation + 15
        ]
        if my-organization = "volunteer" [
          set moral-motivation moral-motivation + 15
        ]
      ]
    ]
  ]
end

;; Create university areas on the map based on real geography
to setup-universities
  ;; Define number of universities
  let num-universities 6  ; All 6 universities to match research data

  ;; List of university names
  let university-names ["NaUKMA" "KNU" "LNU" "KhNU" "ONPU" "ChNU"]

  ;; Regions associated with each university
  let university-regions ["Central" "Central" "West" "East" "South" "Central"]

  ;; University quality factors (0-100) - affects educational influence
  let university-quality [85 75 70 65 60 65]

  ;; Coordinates for each university based on approximate real geography
  let university-coordinates [
    [0 0]     ; NaUKMA - Kyiv (Central)
    [5 -2]    ; KNU - Kyiv (Central)
    [-15 5]   ; LNU - Lviv (West)
    [18 -5]   ; KhNU - Kharkiv (East)
    [5 -15]   ; ONPU - Odesa (South)
    [-5 -10]  ; ChNU - Chernivtsi (Central-West)
  ]

  ;; University size (radius) to create appropriate areas
  let university-sizes [3 6 6 6 6 5]

  ;; Create each university in its geographic location
  let university-counter 0

  repeat num-universities [
    ;; Skip if we've run out of university names
    if university-counter >= length university-names [stop]

    ;; Get the current university coordinates and radius
    let center-x item 0 (item university-counter university-coordinates)
    let center-y item 1 (item university-counter university-coordinates)
    let radius item university-counter university-sizes

    ;; Get the current university name and associated data
    let current-name item university-counter university-names
    let current-region item university-counter university-regions
    let current-quality item university-counter university-quality

    ;; Create a cluster of patches for this university
    ask patches with [(pxcor - center-x) ^ 2 + (pycor - center-y) ^ 2 < radius ^ 2] [
      set university current-name

      ;; Set colors based on university quality (unique shade for each university)
      set pcolor scale-color blue current-quality 0 100
      set plabel-color white

      ;; Occasionally label a patch with university name for visualization
      if random 100 < 10 and (pxcor - center-x) ^ 2 + (pycor - center-y) ^ 2 < (radius - 1) ^ 2 [
        set plabel current-name
      ]
    ]

    ;; Increment counter
    set university-counter university-counter + 1
  ]
end

;; MAIN SIMULATION PROCEDURES
to go
  if ticks >= simulation-length [stop]  ; End simulation after specified time

  ;; Advance war timeline
  set war-week ticks

  ;; Update environmental conditions (media, economy, events)
  update-environment

  ;; Students interact and update motivations
  ask students [
    interact-with-peers
    update-motivations
    decide-to-join
  ]

  ;; Update visualization
  update-display

  tick
end

;; Update environmental conditions affecting all students
to update-environment
  ;; Initialize media influence on first tick
  if ticks = 0 [
    set media-influence 50  ; Start at neutral value
  ]

  ;; Update media influence based on slider and random fluctuations
  set media-influence (media-coverage + random-float 5 - 2.5)
  set media-influence max list 0 min list 100 media-influence  ; Keep within 0-100 range

  ;; Economy gradually fluctuates with war duration
  set economic-hardship max list 0 (economic-hardship + random-float 2 - 1)

  ;; Apply environmental effects to students based on their characteristics
  ask students [
    ;; Recruitment campaigns affect social motivation
    set social-motivation social-motivation +
      (recuitment-campaign-intensity / 100) * social-weight

    ;; Recruitment campaigns also affect economic motivation but less strongly
    set economic-motivation economic-motivation +
      (recuitment-campaign-intensity / 200) * economic-weight

    ;; Eastern and Southern regions face more economic impact during war
    if region = "East" or region = "South" [
      set economic-motivation economic-motivation + (economic-hardship / 200)
    ]

    ;; Media influence affects political motivation differently based on education level
    set political-motivation political-motivation +
      (media-influence / 100) * (educational-influence / 100) * political-weight
  ]

  ;; Occasional major events that affect motivations (roughly every 10 weeks)
  if war-week mod 10 = 0 and war-week > 0 [
    let event-type random 3

    ;; Major military event (increases moral motivation)
    if event-type = 0 [
      set military-events military-events + 1
      ask students [
        set moral-motivation moral-motivation + random-float 10
      ]
    ]

    ;; Economic support package (increases economic motivation)
    if event-type = 1 [
      set economic-events economic-events + 1
      ask students [
        set economic-motivation economic-motivation + random-float 15
      ]
    ]

    ;; Political development (increases political motivation)
    if event-type = 2 [
      set political-events political-events + 1
      ask students [
        set political-motivation political-motivation + random-float 12
      ]
    ]
  ]
end

;; Students interact with peers and update their social network
to interact-with-peers
  ;; Find nearby peers within interaction radius
  let nearby-peers other students in-radius interaction-radius

  ;; Store nearby peers list
  set peers nearby-peers

  ;; Calculate the percentage of peers who have joined military
  ifelse any? nearby-peers [
    set my-peers-joined (count nearby-peers with [joined?] / count nearby-peers) * 100
  ] [
    set my-peers-joined 0  ; No peers, so set to 0
  ]
end

;; Update student motivation levels based on various factors
to update-motivations
  ;; Calculate peer influence
  let joined-peers 0
  let total-peers count peers

  if total-peers > 0 [
    set joined-peers count peers with [joined?]
  ]

  ;; Social motivation increases when peers join
  set social-motivation social-motivation +
    (joined-peers / max list 1 total-peers) * social-weight

  ;; Moral motivation increases with war duration
  set moral-motivation moral-motivation +
    (war-week / 50) * moral-weight

  ;; Political motivation affected by activism involvement
  set political-motivation political-motivation +
    (activism-involvement / 100) * political-weight

  ;; Economic motivation affected by economic conditions
  set economic-motivation economic-motivation +
    (economic-hardship / 100) * economic-weight

  ;; Gender-specific motivation updates based on research
  if gender = "female" [
    ;; Research shows women have higher responsibility for state welfare
    set moral-motivation moral-motivation +
      (war-week / 40) * moral-weight

    ;; Women more motivated to prove themselves against stereotypes
    set social-motivation social-motivation +
      (war-week / 200) * social-weight
  ]

  ;; Apply natural decay to motivations over time (prevents unlimited growth)
  set social-motivation min list 100 (social-motivation * 0.97)
  set moral-motivation min list 100 (moral-motivation * 0.99)
  set political-motivation min list 100 (political-motivation * 0.98)
  set economic-motivation min list 100 (economic-motivation * 0.96)
end

;; Make decision about joining military based on motivations and barriers
to decide-to-join
  if joined? [stop]  ; Skip if already joined

  ;; Calculate overall motivation (weighted sum of motivation types)
  let overall-motivation (
    (social-motivation * social-weight) +
    (moral-motivation * moral-weight) +
    (political-motivation * political-weight) +
    (economic-motivation * economic-weight)
  ) / (social-weight + moral-weight + political-weight + economic-weight)

  ;; Apply barriers based on research findings
  if contract-duration = "3-years" [
    ;; Longer contracts reduce likelihood (significant barrier identified in research)
    set overall-motivation overall-motivation * 0.8
  ]

  if study-service-compatibility < 50 [
    ;; Lower compatibility between studies and service is a barrier
    set overall-motivation overall-motivation * (0.5 + (study-service-compatibility / 100))
  ]

  ;; Make decision to join if motivation exceeds threshold
  if overall-motivation > motivation-threshold [
    set joined? true
    set color red
  ]
end

;; Update display elements for visualization
to update-display
  ;; Update student colors based on joined status
  ask students [
    ifelse joined?
      [ set color red ]     ; Joined military (red)
      [ set color green ]   ; Not joined (green)
  ]
end
@#$#@#$#@
GRAPHICS-WINDOW
210
10
677
388
-1
-1
9.0
1
12
1
1
1
0
1
1
1
-25
25
-20
20
0
0
1
ticks
30.0

BUTTON
8
13
73
47
setup
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
74
12
130
46
go
Go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
17
100
190
133
population-size
population-size
50
500
500.0
50
1
NIL
HORIZONTAL

SLIDER
16
137
189
170
percent-female
percent-female
0
100
50.0
1
1
NIL
HORIZONTAL

SLIDER
17
65
190
98
simulation-length
simulation-length
0
200
200.0
1
1
NIL
HORIZONTAL

SLIDER
16
202
189
235
war-intensity
war-intensity
0
100
70.0
1
1
NIL
HORIZONTAL

SLIDER
16
238
189
271
media-coverage
media-coverage
0
100
50.0
1
1
NIL
HORIZONTAL

SLIDER
10
433
185
466
study-service-compatibility
study-service-compatibility
0
100
30.0
1
1
NIL
HORIZONTAL

SLIDER
16
274
195
307
recuitment-campaign-intensity
recuitment-campaign-intensity
0
100
30.0
1
1
NIL
HORIZONTAL

CHOOSER
10
382
149
427
contract-duration
contract-duration
"1-year" "3-years"
0

SLIDER
211
400
384
433
social-weight
social-weight
0
10
8.0
1
1
NIL
HORIZONTAL

SLIDER
212
437
385
470
moral-weight
moral-weight
0
10
8.0
1
1
NIL
HORIZONTAL

SLIDER
394
400
567
433
political-weight
political-weight
0
10
5.0
1
1
NIL
HORIZONTAL

SLIDER
396
436
569
469
economic-weight
economic-weight
0
10
3.0
1
1
NIL
HORIZONTAL

MONITOR
685
10
782
43
Students Joined (%)
precision (100 * count students with [joined?] / count students) 1
17
1
8

MONITOR
781
10
831
43
Male, %
precision (100 * count students with [gender = \"male\" and joined?] / count students with [gender = \"male\"]) 1
17
1
8

MONITOR
832
10
885
43
Female, %
precision (100 * count students with [gender = \"female\" and joined?] / count students with [gender = \"female\"]) 1
17
1
8

MONITOR
896
10
946
43
Western
precision (100 * count students with [region = \"West\" and joined?] / count students with [region = \"West\"]) 1
17
1
8

MONITOR
947
10
997
43
Central
precision (100 * count students with [region = \"Central\" and joined?] / count students with [region = \"Central\"]) 1
17
1
8

MONITOR
997
10
1047
43
Eastern
precision (100 * count students with [region = \"East\" and joined?] / count students with [region = \"East\"]) 1
17
1
8

MONITOR
1047
10
1097
43
Southern
precision (100 * count students with [region = \"South\" and joined?] / count students with [region = \"South\"]) 1
17
1
8

MONITOR
688
204
738
237
Social
precision mean [social-motivation] of students 1
17
1
8

MONITOR
738
204
788
237
Moral
precision mean [moral-motivation] of students 1
17
1
8

MONITOR
788
205
838
238
Political
precision mean [political-motivation] of students 1
17
1
8

MONITOR
837
206
894
239
Economic
precision mean [economic-motivation] of students 1
17
1
8

PLOT
687
47
1100
197
 Regional Comparison Over Time
Weeks
% Joined by Region
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"West" 1.0 0 -13345367 true "" "plot (100 * count students with [region = \"West\" and joined?] / count students with [region = \"West\"])"
"Central" 1.0 0 -10899396 true "" "plot (100 * count students with [region = \"Central\" and joined?] / count students with [region = \"Central\"])"
"East" 1.0 0 -5825686 true "" "plot (100 * count students with [region = \"East\" and joined?] / count students with [region = \"East\"])"
"South" 1.0 0 -955883 true "" "plot (100 * count students with [region = \"South\" and joined?] / count students with [region = \"South\"])"
"Overall" 1.0 0 -16777216 true "" "plot (100 * count students with [joined?] / count students)"

PLOT
687
240
939
390
Average Motivation Level
Weeks
Average Level (0-100)
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"Social" 1.0 0 -2674135 true "" "plot mean [social-motivation] of students"
"Moral" 1.0 0 -10899396 true "" "plot mean [moral-motivation] of students"
"Political" 1.0 0 -13345367 true "" "plot mean [political-motivation] of students"
"Economic" 1.0 0 -955883 true "" "plot mean [economic-motivation] of students"

PLOT
687
392
943
542
Overall Motivation
NIL
NIL
0.0
101.0
0.0
10.0
true
true
"" ""
PENS
"Motivation" 1.0 1 -7500403 true "" "histogram [((social-motivation * social-weight) + (moral-motivation * moral-weight) + (political-motivation * political-weight) + (economic-motivation * economic-weight)) / (social-weight + moral-weight + political-weight + economic-weight)] of students"
"Threshold" 1.0 0 -2674135 true "" "let mean-threshold mean [motivation-threshold] of students\nplot-pen-up\nplotxy mean-threshold 0\nplot-pen-down\nplotxy mean-threshold plot-y-max"
"-SD" 1.0 0 -13345367 true "" "  let mean-threshold mean [motivation-threshold] of students\n  let sd-threshold standard-deviation [motivation-threshold] of students\n  plot-pen-up\n  plotxy (mean-threshold - sd-threshold) 0\n  plot-pen-down\n  plotxy (mean-threshold - sd-threshold) plot-y-max"
"+SD" 1.0 0 -13345367 true "" "  let mean-threshold mean [motivation-threshold] of students\n  let sd-threshold standard-deviation [motivation-threshold] of students\n  plot-pen-up\n  plotxy (mean-threshold + sd-threshold) 0\n  plot-pen-down\n  plotxy (mean-threshold + sd-threshold) plot-y-max"

SLIDER
15
311
189
344
interaction-radius
interaction-radius
1
10
8.0
1
1
NIL
HORIZONTAL

PLOT
940
240
1140
390
Events
Event Type
Frequency
0.0
0.0
0.0
10.0
true
true
"" ""
PENS
"Military" 1.0 0 -2674135 true "" "plot military-events"
"Economic" 1.0 0 -13840069 true "" "plot economic-events"
"Political" 1.0 0 -13345367 true "" "plot political-events"

TEXTBOX
224
327
361
382
GREEN - Volunteer orgs\nBLUE - Political orgs\nYELLOW - Youth orgs\nRED - joined the army
10
9.9
1

PLOT
945
392
1145
542
Peers Joined
Percentage of Peers Joined
Number of Students
0.0
100.0
0.0
10.0
true
false
"" "clear-plot\nset-histogram-num-bars 10"
PENS
"distribution" 1.0 1 -8431303 true "" "histogram [my-peers-joined] of students"

BUTTON
130
12
186
46
go once
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

@#$#@#$#@
# Student Military Motivation Model

## WHAT IS IT?

This model simulates the decision-making process of students at Kyiv higher education institutions regarding whether to join the defense forces during Russia's full-scale invasion of Ukraine. Based on research by Stratiienko and Prokhorova (2024), the model explores the various factors that motivate students to volunteer for military service, despite being legally exempt from mobilization during their studies.

The model visually represents different universities across Ukraine's regions (visible as blue clusters), along with various organizations (yellow for youth organizations, blue for political organizations, green for volunteer organizations). Students are represented as persons (green if not joined, red if they've joined the military).

The model incorporates four key motivation types identified in the research:
1. Social motivation (finding belonging, peer influence)
2. Moral and psychological motivation (overcoming guilt, showing solidarity)
3. Political motivation (civic duty, historical participation)
4. Economic motivation (financial stability, support for family)

Additionally, the model accounts for background factors that shape these motivations, including family influence, educational environment, and activism involvement.

## HOW IT WORKS

The model represents students as agents with individual attributes (age, gender, study year, region of origin) who are situated within university environments. These students interact with peers and organizations, and their motivations evolve over time based on:

1. **Background factors:**
   - Family influence (varies by region, with Western regions typically having higher Ukrainian-centric values)
   - Educational influence (varies by university quality and region)
   - Activism involvement (participation in youth, political, or volunteer organizations)

2. **Environmental conditions:**
   - War progression (weeks since invasion)
   - Media coverage of military
   - Economic hardship
   - Military, economic, and political events
   - Recruitment campaign intensity
   - Contract duration options
   - Study-service compatibility

3. **Social interactions:**
   - Peer influence (percentage of peers who have joined)
   - Organizational membership

Each student has unique motivation thresholds and values for each motivation type. When their weighted overall motivation exceeds their individual threshold, they decide to join the military.

## HOW TO USE IT

### Setup Parameters
- **Population Size**: Number of students in the simulation (default: 500)
- **Percent Female**: Percentage of female students (default: 50)
- **Simulation Length**: Maximum number of weeks to run the simulation (default: 200)

### Environmental Controls
- **War Intensity**: Overall intensity of the conflict (default: 70)
- **Media Coverage**: Level of positive media representation of military service (default: 50)
- **Recruitment Campaign Intensity**: How aggressively the military is recruiting students (default: 30)
- **Study-Service Compatibility**: How easy it is to combine studies with service (default: 30)
- **Contract Duration**: "1-year" (shorter) or "3-years" (longer) military contracts (default: 1-year)

### Motivation Weight Sliders
- **Social Weight**: Importance of social factors in decision-making (default: 8)
- **Moral Weight**: Importance of moral and psychological factors (default: 8)
- **Political Weight**: Importance of civic and political factors (default: 5)
- **Economic Weight**: Importance of economic factors (default: 3)

### Interaction Controls
- **Interaction Radius**: How far students can interact with peers (default: 8)

### Buttons
- **Setup**: Initialize the model with the current parameter settings
- **Go**: Run the simulation continuously
- **Go Once**: Advance the simulation by one time step (week)

### Monitors and Plots
- **Students Joined (%)**: Overall percentage of students who have joined the military
- **Male, %** and **Female, %**: Breakdown of joining rates by gender
- **Western, Central, Eastern, Southern**: Breakdown of joining rates by region of origin
- **Regional Comparison Over Time**: Graph showing joining rates by region over time
- **Average Motivation Level**: Graph showing the trends of the four motivation types
- **Events**: Graph tracking the frequency of military, economic, and political events
- **Overall Motivation**: Graph showing motivation levels compared to thresholds
- **Peers Joined**: Distribution of peer influence on students

## THINGS TO NOTICE

- How does social influence spread through student networks? Notice how clusters of students tend to join together, visible as red agents concentrated in specific universities.

- Compare the joining rates among students from different regions in the "Regional Comparison Over Time" plot. According to the sample run shown in the interface, Western Ukraine students (blue line) join at higher rates (33.3%) compared to Southern Ukraine (23.1%), aligning with the research findings about higher family influence and Ukrainian-centric values in western regions.

- Observe the differential impact of the four motivation types in the "Average Motivation Level" plot. Social and moral motivations (red and green lines) appear to increase most rapidly and reach the highest levels (100), while political and economic motivations grow more slowly.

- Notice how events (military, economic, political) create sudden shifts in motivation levels, tracked in the "Events" plot.

- Observe the different joining patterns between male and female students (43.4% vs 24.7% in the sample run). Despite higher barriers, female students who do join often have stronger moral motivations.

- Watch how organizational membership (yellow, blue, and green patches) affects motivation. Students in volunteer organizations (green patches) typically develop stronger moral motivations.

## THINGS TO TRY

- Adjust the motivation weights (currently set to Social: 8, Moral: 8, Political: 5, Economic: 3) to see which types of motivation have the strongest effect on joining decisions. Try reversing the weights to see if economic factors could become more influential under different conditions.

- Compare short (1-year, currently selected) versus long (3-year) contract durations to see how commitment length affects volunteering rates. The research indicated that shorter contracts could make joining more appealing to students.

- Vary the study-service compatibility (currently set to 30) to see how making it easier to combine studies with service affects joining rates. Try setting it to values above 50 to model policy changes that better accommodate student-soldiers.

- Change the recruitment campaign intensity (currently 30) and media coverage (currently 50) to see the impact of different outreach strategies. According to the research, targeted campaigns through youth organizations might be particularly effective.

- Modify the interaction radius (currently 8) to explore how social network density affects the spread of joining behavior. Does increasing this value lead to faster social contagion?

- Create scenarios with different war intensity levels (currently 70) to model how escalation or de-escalation of the conflict might influence motivation.

- Run multiple simulations with different percent-female settings (currently 50%) to explore gender dynamics in military volunteerism.

## EXTENDING THE MODEL

- Add a mechanism for students to leave military service and return to studies.

- Implement more detailed university characteristics, including specific academic programs that might influence motivation (e.g., military studies, medicine, engineering).

- Include a representation of family networks, so students can be influenced by relatives in addition to peers.

- Add traumatic events that could significantly affect moral motivation.

- Incorporate international influence factors such as NATO support or geopolitical events.

- Implement a more sophisticated economic model that includes changing military salaries, inflation, and civilian job opportunities.

- Add faculty members as influencers within university environments.

- Include a more detailed representation of different military roles and branches that appeal to different student profiles.

- Create a counterpart to model the decision-making process of students who choose alternative forms of contributing to the war effort (e.g., volunteering, civil defense).

## NETLOGO FEATURES

- The model uses breeds to represent students with complex attributes.

- It uses patches with different properties to represent universities and organizations.

- The model employs agent networks through the "peers" variable to represent social connections.

- It uses scale-color to visually represent university quality on the patches.

- The simulation incorporates probabilistic decisions based on weighted factors.

## RELATED MODELS

- Community models: Rebellion, Ethnocentrism, Segregation
- Social Science models: Paths, Team Assembly, Diffusion on a Directed Network
- Networks models: Preferential Attachment, Virus on a Network, Small Worlds

## CREDITS AND REFERENCES

This model is based on research by Iryna Stratiienko and Anna Prokhorova (2024) titled "The Motives of Students of Kyiv Higher Education Institutions to Serve in Defense Forces After the Full-Scale Invasion of Russia into Ukraine" published in NaUKMA Research Papers Sociology.

The research identified four key categories of motivations among students who volunteered for military service:
1. Social motivations (finding belonging, peer influence)
2. Moral and psychological motivations (overcoming guilt, showing solidarity)
3. Political motivations (civic duty, historical participation)
4. Economic motivations (financial stability, support for family)

The model also incorporates findings on background factors (family upbringing, education, activism) that shape these motivations, as well as regional differences and gender-specific patterns identified in the research.

Additional references from the paper that informed this model include:
- Jonsson et al. (2024) on conscription systems in European countries
- Girsh (2019) on youth attitudes toward military service in Israel
- Klymchuk (2015) on motivational discourse of personality
- Klymeniuk & Vakhovska (2018) on gender aspects of military service motivation
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.4.0
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
