extensions [table]
globals[stores-traffic calculated?]
breed [customers customer]
breed [stores store] ;; Declare store breed after customer
;; breed so that stores will be viewed on top of customers.

stores-own [store-size]
customers-own [store-distances store-probabilities 
  total-distance
  total-attractiveness
  store-attractivenesses]

to setup
  clear-all
  set calculated? false
  ask patches [set pcolor white]
  create-customers num-customers
  ask customers [setxy random-xcor random-ycor
    set color gray
    set shape "circle" 
    set size 1 / 2
    ]
  
end

to add-drop-stores
  let erasing? any? stores-on patch mouse-xcor mouse-ycor
  while [mouse-down?]
    [ask patch mouse-xcor mouse-ycor
      [ifelse erasing?
        [remove-store]
        [add-store]]
      display]
end

to remove-store
  ask stores-here [die]
end

to add-store
  if not any? stores-here
  [ask stores-here [die] ;; any existing stores are removed
    sprout-stores 1 [set color black
      set shape "house"
      set size 2
      set store-size 1]]
end

to calculate-attractivness-statistics
  let n count stores
  ask customers [set store-distances []
    set store-probabilities []
    set store-attractivenesses []]  
  ; Record the store distances for each turtle.
  ask customers [
    let the-turtle self
    ask stores [
      let the-distance distance the-turtle
      let the-store self
      ask the-turtle [set store-distances fput (list the-distance the-store) store-distances]
    ]
  ]
  ; Not really needed:
  ask customers [set store-distances sort-by [first ?1 < first ?2] store-distances]
  ; Record the store-attractivenesses for each turtle.
  ; The attractiveness of a store is the
  ; product of the size of the store powered
  ; to the value of alpha, times the
  ; reciprocal of 
  ; distance to that store, powered to beta.
  ask customers [set store-attractivenesses table:make
    set total-distance  total-distances
    let td total-distance    
    foreach store-distances [
     let dist first ?
     let store last ?
     table:put store-attractivenesses [who] of store ;(list [pxcor] of store [pycor] of store) 
        ([store-size] of store) ^ alpha * (1 / dist) ^ beta
    ]
  ]
  ; Record the store-probabilities for each turtle.
  ask customers [set store-probabilities table:make
    set total-attractiveness total-attractivenesses
    foreach table:keys store-attractivenesses [
     table:put store-probabilities ? 
     (table:get store-attractivenesses ? / total-attractivenesses)
     ]
    ]
end

to-report total-distances
  let answer 0
  foreach store-distances
  [set answer (answer + first ?)]
  report answer
end

to-report total-attractivenesses
  let answer 0
  foreach table:keys store-attractivenesses
  [set answer (answer + table:get store-attractivenesses ?)]
  report answer  
end

to calculate-store-traffic
  calculate-attractivness-statistics
  set stores-traffic table:make
  ask stores [  ; let x [xcor] of self
   ;let y [ycor] of self
   let key [who] of self ;(list x y)
   table:put stores-traffic key 0
  ]
;  set traffic-accumulator 0
  let keys table:keys stores-traffic
  foreach keys [
   ask customers [table:put stores-traffic ? 
     (table:get stores-traffic ? + table:get store-probabilities ?)
  ]
  ]
  ; ask stores [set expected-traffic table:get stores-traffic who]
end
to display-results
  ; Now display it
  
  output-print (word "Store           Expected")
  output-print (word "  x   y    ID   Traffic")
  let listtable table:to-list stores-traffic
  set listtable sort-by [first ?1 < first ?2] listtable
  
  let sep ""
  repeat 24 [set sep (word sep "-")]
  let sepstar ""
  repeat 24 [set sepstar (word sepstar "*")]
  output-print sep
  foreach listtable [
    let x (word [xcor] of store first ?)
    let y (word [ycor] of store first ?)
    let id (word first ?)
    let paddingx ""
    repeat (3 - length x) [set paddingx (word paddingx  " ")]
    let paddingy ""
    repeat (4 - length y) [set paddingy (word paddingy  " ")]
    let paddingid ""
    repeat (6 - length id) [set paddingid (word paddingid " ")]
    let the-traffic (round (100 * last ?) / 100)
    let paddingtraffic ""
    repeat (9 - length (word the-traffic)) [set paddingtraffic (word paddingtraffic " ")]
    output-print (word  paddingx x paddingy y   paddingid id
      paddingtraffic the-traffic)]
  output-print sepstar
end

to explore
  no-display
  let x xcor
  let y ycor
  let stores-traffic-temp stores-traffic
  move-to one-of patches in-radius exp-radius
  calculate-store-traffic
  if table:get stores-traffic who < 
       table:get stores-traffic-temp who
  ;; The move is not an improvement, so reverse it.
  [;print "No improvement"
    set xcor x
   set ycor y
   set stores-traffic stores-traffic-temp]
  display
end

to-report get-store-ids
  let ids []
  ask stores [set ids fput who ids]
  report ids
end
@#$#@#$#@
GRAPHICS-WINDOW
207
10
654
478
16
16
13.242424242424242
1
10
1
1
1
0
0
0
1
-16
16
-16
16
0
0
1
ticks
30.0

BUTTON
73
10
211
43
NIL
add-drop-stores
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
7
43
210
76
num-customers
num-customers
0
1000
1000
10
1
NIL
HORIZONTAL

BUTTON
7
10
73
43
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
7
76
210
109
alpha
alpha
0.1
5.0
2
0.1
1
NIL
HORIZONTAL

SLIDER
7
109
210
142
beta
beta
0.1
5
2
0.1
1
NIL
HORIZONTAL

OUTPUT
2
313
206
478
12

BUTTON
7
142
211
175
Calculate Traffic & Display Results
calculate-store-traffic\nset calculated? true\nclear-output\ndisplay-results
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
6
280
85
313
Explore
ifelse calculated? = false\n[user-message \"Cannot be done until results are calculated.\"]\n[ask stores [explore]]\ntick
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

INPUTBOX
7
175
112
235
exploring-store-id
1001
1
0
Number

BUTTON
108
239
211
283
Store ID Explore
ifelse calculated? = false\n[user-message \"Cannot be done until results are calculated.\"]\n[let ids get-store-ids\nifelse member? exploring-store-id  ids\n[ask store exploring-store-id [explore]\ntick]\n[user-message (word exploring-store-id \n\" is not a valid store ID. \" \n\"The valid IDs are: \" sort ids \".\")]\n]\n
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

MONITOR
7
235
108
280
Store Exp. Traffic
table:get stores-traffic exploring-store-id
2
1
11

BUTTON
113
207
211
240
Display results
ifelse calculated? = false\n[user-message \"Cannot be done until results are calculated.\"]\n[display-results]
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
85
280
211
313
exp-radius
exp-radius
1
10
4
1
1
NIL
HORIZONTAL

BUTTON
112
174
211
207
NIL
reset-ticks
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

@#$#@#$#@
## WHAT IS IT?

A number of  establishments that compete for regional customer exist or are contemplated in a particular  geographic area.  Given the several locations and attractivenesses of these establishments, we would like to predict the market share each will achieve.  We are interested in deciding  whether to build a number of new establishments as well as where to put any we do decide to build.

This problem falls under the category of **market area analysis**, addressing one of the category's main concerns: the location of prospective retail stores. The Huff model is used extensively for this purpose. The Huff NetLogo model constitutes a simple implementation of the Huff model.

The basic formula for the Huff model is

P(i,j) = (S(j)<sup>alpha</sup> / T(i,j)<sup>beta</sup>) / sum(i=1,n)[(S(j)<sup>alpha</sup> / T(i,j)<sup>beta</sup>)]

where

* P(i,j) is the probability of a consumer at a given origin _i_ traveling to a particular shopping center _j_.
* S(i) is the size of a shopping center _j_ (typically measured in terms of the square footage of selling area devoted to the sale of a particular class of goods). More generally, a measure of attractiveness of the shopping center (with higher values more attractive than lower values).
* T(ij) is the travel time or cost for customer  _i_ to go to shopping center _j_.

* alpha is a parameter, to be estimated empirically but often simply set to 1, for scaling the importance of the attractiveness measure for the establishment, S(j).
* beta is a parameter, to be estimated empirically but often simply set to 2 (hence the literature often speaks of the  "Huff **gravity** model" for this class of models), reflecting the effect of travel time and other costs on various kinds of shopping trips.
* _n_ is the number of establishments or shopping centers under consideration.

Further, it is presumed in using this model that there are _m_ customers, that each customer has a known location, and that each of the establishments under consideration has a known location, real or hypothetical.

The expected traffic for a store _j_ is

expected_traffic(j) = sum(i=1,m)[P(i,j)]


## HOW IT WORKS

Customers are represented by turtles of breed customers and displayed as small gray circles. The setup procedure arrays them randomly. There is nothing in the implementation that makes them move.  Stores are represented by turtles of breed stores. The calculate-store-traffic procedure calculates P(i,j) for all customers _i_ and stores _j_. The display-results procedure reports to the output widget the expected traffic for each store in the model. These two procedures are combined, with some error checking, in the Calculate Traffic & Display Results button.

You make pick an individual store and direct it to explore for an improved position. At each tick the chosen store will pick a random patch in its neighborhood of radius exp-radius, travel to the patch and cause the system to re-evaluate everyone's expected traffic. If the store finds its new expected traffic is better, it remains where it is and the tick is over. If not, the store returns to its original position and causes the original expected traffic values to be restored. You can direct this with the Store ID Explore button, which is a forever button.

Alternatively, you may ask all of the stores to explore in parallel. This is accomplished essentially with an `ask stores` directive, instead of an `ask store X` directive, where X is a store ID.  You direct this with the Explore button, which is also a forever button.

## HOW TO USE IT

Use the sliders to set the number of customers, _n_ in the Huff formula, and alpha and beta in the Huff formula. Click the setup button to initialize the world. To add  stores, click the add-drop-stores button and then click in the world on any patch that does not already have a store. If your click on a patch that already has a store over it, the store gets removed. Click the add-drop-stores button again when you are done adding or removing stores. Click on the Calculate Traffic & Display Results button  to run the Huff formula on each of the customer-store combinations, and to report in the output widget the expected traffic for each store. Stores have a default size of 1 when they are added. 

Once you have clicked on the Calculate Traffic & Display Results button you may direct individual or collective search for improved positions. To direct individual search by store with ID X,

1. Edit the input widget exploring-store-id so that it contains X.
2. Click the reset-ticks button.
3. Adjust the exp-radius slider to have the value you want.
4. Click the Store ID Explore button.
5. Watch while the store moves around and the tick counter increases. 
6. Click the Store ID Explore button to halt search when you wish.
7. Click the Display results button.

To direct collect search for improved positions,

1. Click the reset-ticks button. This starts our clock anew.
2. Adjust the exp-radius slider to have the value you want.
3. Click the Explore button.
4. Watch the stores move as the ticks counter increases.
5. Click the Explore button again to halt search when you wish.
6. Click the Display results button.



## THINGS TO NOTICE

Changing the size of a store can have a substantial affect on the level of its expected traffic, especially if alpha is much greater than 1. You can easily change the size of a store by inspecting it and editing the size attribute.

The payoffs resulting from changes in position are often surprising, usually interesting.

## THINGS TO TRY

You can use NetLogo commands to move the turtles/customers to non-random distributions. For example:

    observer> ask turtles [set heading 180]
    observer> ask turtles [fd 5]

Again, the payoffs resulting from changes in position are often surprising, usually interesting.


## NETLOGO FEATURES

The code uses the NetLogo table extension multiple times in its calculation of P(i,j) and expected traffic.

## RELATED MODELS

The Huff Turtles-Based NetLogo model is an extension of the Huff NetLogo model.

The coding methods here draw upon the Life NetLogo model, found in the Models Library.

The Huff model has an interesting relationship with Hotelling's spatial location models, which are implemented in the Hotelling's Law NetLogo model in the Models Library.

## HOW TO CITE

If you mention this model in a publication, we ask that you include these citations for the model itself and for the NetLogo software:

* Kimbrough, Steven O. (2014). Huff model.  University of Pennsylvania, Philadelphia, PA. Huff.nlogo

## COPYRIGHT AND LICENSE

Copyright 2014 Steven O. Kimbrough.

![CC BY-NC-SA 3.0](http://i.creativecommons.org/l/by-nc-sa/3.0/88x31.png)

This work is licensed under the Creative Commons Attribution-NonCommercial-ShareAlike 3.0 License.  To view a copy of this license, visit http://creativecommons.org/licenses/by-nc-sa/3.0/ or send a letter to Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.

Commercial licenses are also available. To inquire about commercial licenses, please contact Steven O. Kimbrough at kimbrough@wharton.upenn.edu.

Version: $Id: Huff Turtle-Based.nlogo 4336 2014-09-08 21:12:50Z sok $.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.1.0
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
