
;;;;;;;;;;;;;;;;;;;;;;;;;;;
; globals and definitions ;
;;;;;;;;;;;;;;;;;;;;;;;;;;;

breed [volunteers volunteer]
volunteers-own [ v_motivation xhome yhome tasking-distance ]

breed [issues issue]
; i_resolved? is set to true when we have enough reports.
issues-own [ i_tasked? i_resolved? i_generated i_tasked i_resolved ]

directed-link-breed [task-requests task-request] ; task requests are special links from the manager to the issue.
                                                 ; task requests are used to generate tasks later on.
task-requests-own [ t_manager-id t_generated-at t_resolved? t_x1 t_x2 t_y1 t_y2 t_outstanding t_issue-ID]
  ; t_outstanding is set to number of requested confirmations initially, and one is substrqcted every time a new volunteer accepts the task
  ; t_resolved is set to true once all the v_tasks generated by this task request have been resolved. 

directed-link-breed [ v_tasks v_task ] ; these are the actual tasks - links from volunteers to issues
directed-link-breed [ observations observation ] ; report is reserved word, so we'll use observation.


; not sure what we need in this link...
observations-own [ timestamp correct?]
; every task is requested by one manager and has a task ID. 
; this is not unique! Same manager + task ID combination is shared with Nr-volunters other v_tasks!
v_tasks-own [ manager-ID task-ID ]

;;;;;;;;;;;;;;;;;;;
; HUBNET STUFF!!! ;
;;;;;;;;;;;;;;;;;;;

;; create a breed of turtles that represents the crisis manager(s). 
;; Crisis managers manage the tasks - in this model the manager will be a human and volunteers software agents.
breed [ managers manager ]

managers-own
[
  user-id          ;; Managers choose a user name when they log in.
                   ;; Each hubnet message from a manager will contain this ID as hubnet-message-source
  x1 y1            ;; one vertice of the BB
  x2 y2            ;; second vertice
  Task-type
  Issue-ID
  Nr-confirmations ;; number of requested volunteers
  Window-size      ;; size of the world-window shown to manager
  ;; more to come
]

;; the STARTUP procedure runs only once at the beginning of the model
;; at this point we just initialize the hubnet
to startup
  hubnet-reset
end

;;
;; HubNet Procedures
;;


to hn-setup-managers
  ask managers
  [
    set Nr-confirmations 3
    hn-inform-me "Nr-confirmations" 3
    set x1 random Size-World - 20
    set y1 random Size-World - 20
    set x2 x1 + 10
    set y2 y1 + 10
    set Task-type "resolve-issue"            ;; right now we have only one type
    set Window-size floor ( Size-world / 5 ) ;; 
    ;; set Issue-ID [who] of one-of issues
    ;; hubnet-send user-id "step-size" step-size
    hn-update-me
  ]
end
  

;; when a new user logs in create a manager turtle
;; this turtle will store any state on the client
;; values of sliders, etc.
to hn-create-new-manager
  create-managers 1
  [
    ;; This is for housekeeping purpose. We need to know who is who.
    set user-id hubnet-message-source
    set label user-id

    set Nr-confirmations 3
    set x1 random Size-World - 20
    set y1 random Size-World - 20
    set x2 x1 + 10
    set y2 y1 + 10
    set Task-type "resolve-issue"
    set Issue-ID [who] of one-of issues
    
    ;; update the clients with any information you have set
    hn-update-me
  ]
end

;; when a user logs out make sure to clean up the turtle
;; that was associated with that user (so you don't try to
;; send messages to it after it is gone) also if any other
;; turtles of variables reference this turtle make sure to clean
;; up those references too.
to hn-remove-manager
  ask managers with [user-id = hubnet-message-source]
  [ die ]
end 

to hn-process-clients
  ;; as long as there are more messages from the clients
  ;; keep processing them.
  while [ hubnet-message-waiting? ]
  [
    ;; get the first message in the queue
    hubnet-fetch-message
    if hubnet-enter-message? [ hn-create-new-manager stop ] ;; when clients enter we get a special message
    if hubnet-exit-message?  [ hn-remove-manager     stop ] ;; when clients exit we get a special message
    print (word "new hubnet message " hubnet-message-tag)
    ask managers with [user-id = hubnet-message-source]
      [ hn-execute-command hubnet-message-tag ] 
    ;; I want to always center on currently chosen issue
    if (hubnet-message-tag = "Prev-Issue" or hubnet-message-tag = "Next-Issue") [
      let ID [Issue-ID] of one-of managers with [user-id = hubnet-message-source]
      hubnet-send-watch hubnet-message-source issue ID
      ;; getting the system to center on one of issues is not as simple as I expected. :-(
      ;; it also complicates the definition of the task area because of the wrapping.
      ;; for now we'll therefore stay with full view
      ;hubnet-send-follow hubnet-message-source issue ID 10
    ] 
    if (hubnet-message-tag = "Mouse Up") [
      ;; context can be quite an annoying topic in netlogo. :-(
      let px1 0 let py1 0
      let px2 0 let py2 0
      ask one-of managers with [user-id = hubnet-message-source] [
        set px1 x1 set py1 y1
        set px2 x2 set py2 y2
      ]
      hn-fill-tasking-area blue px1 py1 px2 py2       ;; this will make the tasking area visible 
    ]
  ]
end


;; Other messages correspond to users manipulating the
;; client interface, handle these individually.
to hn-execute-command [command]
  ;; you should have one if statement for each widget that
  ;; can affect the outcome of the model, buttons, sliders, switches
  ;; choosers and the view, if the user clicks on the view you will receive
  ;; a message with the tag "View" and the hubnet-message will be a
  ;; two item list of the coordinates
  print (word "command=" command "(" hubnet-message ")") 

  ;;if command = "Task-type" [ set Task-type item 0 hubnet-message stop ]
  if command = "Nr-confirmations" [ set Nr-confirmations hubnet-message stop ]
  if command = "Nr-confirmations" [ set Nr-confirmations hubnet-message stop ]
  if command = "Task-nearest?"    [ set Task-nearest? hubnet-message stop ]

  ;; find next unresolved issue which is eligible for tasking.
  if command = "Next-Issue" [
    let next_issue min-one-of ( issues with [ who > [Issue-ID] of myself and not i_tasked?  ] ) [ who ]
    if not is-issue? next_issue [ set next_issue min-one-of issues with [ not i_tasked?  ] [ who ] ]
    print (word "Issue-ID=" Issue-ID " next-issue=" next_issue)
    ifelse is-issue? next_issue [
      set Issue-ID [who] of next_issue
      hn-inform-me "Issue-ID" Issue-ID        
      set color blue 
      ; manager "moves" to the issue.
      set xcor [xcor] of next_issue
      set ycor [ycor] of next_issue
      stop
    ] 
    [
      ;; TODO: if this does not work either, there are no issues left. What now? 
    ]   
    stop
  ]
  ;; find previous unresolved issue which is eligible for tasking.
  if command = "Prev-Issue" [
    let prev_issue max-one-of ( issues with [ who < [Issue-ID] of myself and not i_tasked?  ] ) [ who ]
    if not is-issue? prev_issue [ set prev_issue max-one-of issues with [ not i_tasked?  ] [ who ] ]
    ifelse is-issue? prev_issue [
      set Issue-ID [who] of prev_issue
      hn-inform-me "Issue-ID" Issue-ID        
      set color blue 
      print (word "Issue-ID=" Issue-ID " prev-issue=" prev_issue)
      ; manager "moves" to the issue.
      set xcor [xcor] of prev_issue
      set ycor [ycor] of prev_issue
      stop
    ]
    [
      ;; TODO: if this does not work either, there are no issues left. What now? 
    ]   
    stop
  ]

  ;; Tasking area is chosen by mouse klick ("View"), drag and letting go ("Mouse up") 
  if command = "View"
  [
    set x1 floor item 0 hubnet-message
    set y1 floor item 1 hubnet-message 
    hn-inform-me "x1" x1
    hn-inform-me "y1" y1
    print (word "x1=" x1 ", y1=" y1) ;; debug
    stop
  ]
  if command = "Mouse Up"
  [
    set x2 floor item 0 hubnet-message
    set y2 floor item 1 hubnet-message 
    hn-inform-me "x2" x2
    hn-inform-me "y2" y2
    print (word "x2=" x2 ", y2=" y2) ;; debug
    stop
  ]
  
  ;; And now it gets serious. We have to add a new task request to the queue.
  if command = "Submit"
  [
    print (word "Time to do the tasking to issue " Issue-ID )
    hn-gen-task-request
    stop
  ]
  
end

;; generate a new tasking request. Called after pressing "Submit" buton.
;; Note: for now this must be called in manager context.
to hn-gen-task-request
  ;; This should assure we never sending the task to same issue twice.
  if [i_tasked?] of issue Issue-ID [ stop ]
  create-task-request-to issue Issue-ID [ 
    set t_manager-id [user-id] of myself
    set t_generated-at ticks
    set t_x1 [x1] of myself ; coordinates of the tasking area
    set t_x2 [x2] of myself
    set t_y1 [y1] of myself
    set t_y2 [y2] of myself
    set t_outstanding [Nr-confirmations] of myself
    set t_issue-ID [Issue-ID] of myself
    set color black
  ]
  ask issue Issue-ID [
    set i_tasked? true
    set i_tasked ticks
  ]
end


;; whenever something in world changes that should be displayed in
;; a monitor on the client send the information back to the client
to hn-update-me ;; turtle procedure
  hubnet-send user-id "x1" x1
  hubnet-send user-id "y1" y1
end

to hn-inform-me [variable value] ;; turtle procedure
  hubnet-send user-id variable value
end

;; make the current tasking area visible for manager
to hn-fill-tasking-area [the_color px1 py1 px2 py2 ]
  ask patches [ 
    ifelse ( pxcor >= min (list px1 px2)
      and pxcor <= max (list px1 px2)
      and pycor >= min (list py1 py2) 
      and pycor <= max (list py1 py2))
    [ set pcolor the_color ] 
    [ set pcolor grey + 4 ]
  ]
end
  


;;;;;;;;;;;;;;;;;;;; 
; setup procedures ;
;;;;;;;;;;;;;;;;;;;;

to setup
  ; clear-all - this would also kill the manager. no-no.
  clear-patches
  clear-drawing
  clear-output
  clear-all-plots
  ask turtles with [breed != managers] [die]
  hn-setup-managers
  setup-world
  setup-volunteers
  clear-ticks
  reset-ticks
end


; initially all patches are just grey = unknown.
to setup-world
  resize-world 0 Size-world 0 Size-world ; world can be of different sizes
  set-patch-size 1000 / Size-world       ; but the world illustration is always the same size. 
  ask patches [
      set pcolor grey + 4
  ]
end


; randomly add some taskable volunteers to the world
to setup-volunteers
  set-default-shape volunteers "person"
  create-volunteers Nr-volunteers [ 
    setxy random-xcor random-ycor
    set color green                  ; green indicates the volunteer is free and ready to get a task.
    set size 1

    ; Variables below aren't used yet (code not implemented)
    set xhome xcor                   ; Just in case we want to send volunteers back home again. 
    set yhome ycor 
    set v_motivation init-motivation ; motivation is not used at the moment either!
  ]
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Procedures governing the world development ;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to go 
  every Issue-period [ add-issues 1 ] ; new issues are added every Issue-period sec.
  every 0.1 [ hn-process-clients ] ;; every 0.1 sec should be more than enough?
  every Issue-period [ export-world (word "volounteers-WS-" ticks)]
  assign-tasks
  move-volunteers
  report-issues
  manage-issues
  tick
end

; adds N (normally just one) new issues to the world when called
; issues have a generation time & later also a discovery time

to add-issues [ N ]
  set-default-shape issues "target"
  create-issues N [ 
    setxy random-xcor random-ycor
    set i_generated ticks            ; this could be used to generate issues as we go.
    set i_resolved? false
    set i_tasked? false
    set color red
    set size 1
  ]
end

; task requests are defined by manager, but we still have to assign them to volunteers 
; since the tasking area may not have enough volunteers at first, 
; we need to work with queue.
to assign-tasks
  ;; vol,unteer can't have >1 tasks at any given moment.
  let free_volunteers volunteers with [not any? out-v_task-neighbors]
  ;; t_outstanding will be zero if we have enough volunteers set on a task.
  ask task-requests with [t_outstanding > 0 ] [
    let this_ID t_issue-ID
    ask ifelse-value Task-nearest? [
      min-n-of t_outstanding free_volunteers [ distance (issue this_ID) ]
    ] [
    n-of t_outstanding free_volunteers with [ 
      xcor >= [min (list t_x1 t_x2)] of myself and 
        xcor <= [max (list t_x1 t_x2)] of myself and 
        ycor >= [min (list t_y1 t_y2)] of myself and 
        ycor <= [max (list t_y1 t_y2)] of myself ]
    ]
    [
      ;; TODO: check if this assures we don't task too many people!
      if [t_outstanding] of myself > 0 [
        set tasking-distance distance issue this_ID ; store the tasking distance of latest task in volunteers variable.
        create-v_task-to issue this_ID [
          if show-tasks? [ set color orange ]
          if debug? [ print word "created task:" self ]
          ;; back to volunters context.
          ;; lets make the tesked v. better visible!        

          ask myself [         ;; myself = volunteer here!
            set color orange
            set size size + 1
          ]
        ]
        ask myself [ ;; one outstanding task-request less to go!
                     ;; myself = task request here
          set t_outstanding t_outstanding - 1
        ]
      ]
    ]
  ]
end


;; volunters will move randomly if not tasked and towards the issue if tasked
to move-volunteers
  ask volunteers [
    ; current target?
    ; normally there should be only one, so "one-of" is the same as
    ; 

    ; If we have a task, let's go for it!
    ifelse any? out-v_task-neighbors [
      face min-one-of (out-v_task-neighbors)  [distance myself ] ;; this will work even if we set several tasks on the volunteer at once.   
      forward 1
    ]
    ; else we'll stroll arround
    [
      right random 360 
    ] 
    forward 1
    ; motivation always goes down, unless an issue is discovered
    ; TODO: this is not implemented yet
    ;if v_motivation > 0 [
    ;  set v_motivation v_motivation - 1
    ;]

    ;; othervise we'll do nothing.
  ]
end


; tasked volunteers move towards their task or at random if no task defined.


; report observations on nearby issues
to report-issues
  ask volunteers with [any? out-v_task-neighbors] [ ;; in this model ONLY the tasked volunteers can send reports.
    ;;let this_volunteer self
    ask one-of out-v_task-neighbors [      ;; that would be the relevant issue
                                           ;; there can be only one since we allow one task per volunteer    
      if distance myself <= R-discover [;; if the issue is near enough, let's report.
                                           ;; we are in issue context
                                           ;; issue = self, volunteer = myself
        create-observation-from myself [   ;; volunteer should report (create) an observation to this issue 
                                           ;; now we are in onservation (link) context!
          set timestamp ticks  ;; when was the observation made?
          if show-observations? [ 
            set color green ;; this will make the observation link visible
          ] 
          if debug? [print self]
        ]
        ask myself [                    ;; this should be the volunteer, we are back in issue context.
          set color green               ;; volunteer is free to take a new task again
          set size size - 1             ;; Disable this to see good reporters growing.
          ask one-of my-out-v_tasks [die]  ;; delete the task from a caling turtle to this issue
        ]
      ]
    ]
  ]
end

to manage-issues 
  ask task-requests with [ t_outstanding <= 0 ] [ ;; should never be < 0, but just in case.
    ;; t_outstanding = 0 means we have tasked enough v.
    ask issue t_issue-ID [
      if not any? in-v_task-neighbors [ ;; if no tasks are left, all volunteers heve reported 
        set i_resolved? true
        set i_resolved ticks
        set color black
        ; ask myself [die] ;; this would kill the task request.
      ]
    ]
  ]
end
@#$#@#$#@
GRAPHICS-WINDOW
267
13
1297
1064
-1
-1
20.0
1
12
1
1
1
0
1
1
1
0
50
0
50
1
1
1
ticks
60.0

BUTTON
11
16
80
59
Reset
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
99
16
167
59
Run
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

BUTTON
187
14
256
59
Step
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

SWITCH
10
856
160
889
debug?
debug?
0
1
-1000

MONITOR
194
152
262
205
Resolved
count issues with [ i_resolved? ]
0
1
13

MONITOR
101
152
185
205
Discoverred
count issues with [i_tasked?]
0
1
13

SLIDER
8
351
260
384
Nr-volunteers
Nr-volunteers
0
500
100
50
1
NIL
HORIZONTAL

MONITOR
10
400
123
453
Free volunteers
count volunteers with [not any? out-v_task-neighbors]
0
1
13

SLIDER
14
1015
262
1048
init-motivation
init-motivation
0
10
3
1
1
NIL
HORIZONTAL

SLIDER
7
100
254
133
Issue-Period
Issue-Period
10
240
10
10
1
sec
HORIZONTAL

TEXTBOX
10
330
244
348
Taskable volunteers (crowdtasking)
13
0.0
1

SWITCH
10
932
160
965
show-tasks?
show-tasks?
0
1
-1000

SWITCH
11
894
160
927
show-observations?
show-observations?
1
1
-1000

SLIDER
8
235
259
268
Size-world
Size-world
50
200
50
10
1
patches
HORIZONTAL

TEXTBOX
29
994
241
1026
motivation is not implemented.
13
0.0
1

TEXTBOX
9
213
159
231
Various dimensions
13
0.0
1

TEXTBOX
13
831
163
849
More info & debugging
13
0.0
1

MONITOR
131
400
251
453
Tasked volunteers
count volunteers with [any? out-v_task-neighbors]
0
1
13

TEXTBOX
11
80
201
98
New issue generation period
13
0.0
1

MONITOR
10
151
91
204
Generated
count issues
0
1
13

SLIDER
8
283
158
316
R-discover
R-discover
1
10
3
1
1
patches
HORIZONTAL

PLOT
10
524
250
815
plot 1
time
tasking distance
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot mean [tasking-distance] of volunteers"

SWITCH
12
473
162
506
Task-nearest?
Task-nearest?
0
1
-1000

@#$#@#$#@
## WHAT IS IT?
This model is a proof of concept for a crowdtasking integration tool. It shows how to realize a client-server application which allows a human operator ("manager") to define task requests and ask for the volunteers. The volunteers are modelled as (rather stupid) agents who will always accept and execute such tasks if they are free to do so.

## HOW IT WORKS

Manager is a human, and interacts with the model using the HubNet interface. Manager (turtle breed) can define a **tasking area** by clicking and dragging a mouse on the map, choose one of the **issues**, decide how many volunteers should be tasked and post a **tasking request** (links from manager to issues). Alternatively, the manager can use the "Task-nearest?" switch to signal the system that the nearest free volunteer should be tasked.

**Tasking requests**  are queued. 

Volunteers walk randomly around unless they are tasked. As soon as a free volunteer arrives to a tasking area, it receives a task. This is repeated until the requested number of volunteers has received (and accepted) the task. Motivation and free will play no role in this model. If the "Task-nearest?" switch is on, the tasking area is ignored and the task is given to volunteers nearest to the issue.

Once they receive a task, volunteers will head straight towards the issue which they need to report on. This model does not implement roads or terrain modifiers.

Once the volunteer is near the issue, it will report an observation and the task will be resolved. 

## HOW TO USE IT

On the "server" side, it is possible to adjust the world size and the number of volunteers, as well as two more subtle parameters

  * **Reset**, **Run** and **Step 1** buttons have the usual meanings.
  * **Nr-issues** determines the number of issues which need to be discovered.
  * **Size-world** determines the size of the world in patches (both x and y dimension)
  * **R-discover** determines the distance from which the issues can be spotted by 
  * **Nr. Volunteers** allows adjusting the number of volunteers.

On the client side, it is possible to jump between "open" issues, choose a tasking area and change the number of volunteers which will receive each of the tasks.

"Task-nearest?" switch has the same meaning on client and on server side. See "How it works" section.

## THINGS TO NOTICE

Combination of "many issues", "distinct tasking areas" and "many volunteers" will quickly discourage even the best manager. This is exactly the type of issues the model should expose in integration tests. Using the "Task-nearest?" option means less work for the manager and the tasking is usually more efficient in terms of the distance the volunteers have to walk and the time needed to finalize the tasks.


## THINGS TO TRY

Observe the difference between "using the same tasking area all the time" and "choosing the area for each issue". 

Observe the difference in average "tasking distance" with and without "Task-nearest?" option.

## EXTENDING THE MODEL

### MOTIVATION & EFICIENCY
In this model it is OK to choose the whole world as the tasking area and randomly assign tasks to some volunteers. This is awfully inefficient and also a great way to discourage volunteers:

  1) if everyone gets all task requests, our volunteers will quickly will flip out (spamming).
  2) If the task sends volunteers far-away, they will spend most time walking. Frustration is sure to happen.

There is more to motivation than this, e.g. thanking the volunteers, making their results visible ("hall of fame"), we could even have some monetary compensations. 

"Task-nearest?" option makes the tasking more efficient, but further optimization could be achieved if the volunteers are able to look ahead and choose several nearby tasks to resolve at once.

### REAL WORLD CONSIDERATIONS

In the real world, people live in houses, work in offices and usually walk or drive on the roads. Their movement is also constrained by topography. 

On top of this, they move at certain speeds (e.g. 4-5km/h for walking), eat, sleep, etc. Eventually, at least some of these features need to be modelled.

### Observation content, errors and quality assurance
In this model, the volunteers will never make an error and return false observation. In fact, there is no content in "observations" at all. 

### TRUST and other attributes
In this model, there is no difference between volunteers. In reality, we will assign difficult tasks to people who are trusted and able to perform them and less critical tasks to a mass of untrusted volunteers. 

In addition, some tasks will require special skills or tools - this needs to be introduced in the volunteer data model.

### MORE TASK TYPES

We can come far by adding a few parameters (e.g. "time to finish this task" nor "possible observation responses") to "issues", but there is one notable exception: this model can't simulate the volunteers response to generic requests such as "visit your neighbour and report weather a medical assistance is needed.".

## NETLOGO FEATURES

  * Uses "breeds" to differ between various types of agents.
  * Uses links to model the task requests, tasks and observations.
  * Custom shapes, sizes and colours of various turtle breeds.
  * HubNet code
  * Mouse events
  * Time events

## RELATED MODELS

See other "Volunteers" models by the same author.

## HOW TO CITE

If you mention this model in a publication, we ask that you include these citations for the model itself and for the NetLogo software:

  * Havlik, D. (2015).  Managing the Volunteers v1 NetLogo model and HubNet activity. http://modelingcommons.org/browse/one_model/4265
  * Wilensky, U. (1999). NetLogo. http://ccl.northwestern.edu/netlogo/. Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.

## COPYRIGHT AND LICENSE

Copyright 2015 Denis Havlik

![CC BY-NC-SA 3.0](http://i.creativecommons.org/l/by-nc-sa/3.0/88x31.png)

This work is licensed under the Creative Commons Attribution-NonCommercial-ShareAlike 3.0 License.  To view a copy of this license, visit http://creativecommons.org/licenses/by-nc-sa/3.0/ or send a letter to Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.

To inquire about commercial licenses and model extensions, please contact Denis Havlik at denis@havlik.org.

## CREDITS AND REFERENCES

This model is an indirect consequence of my work in ENVIROFI and DRIVER EU projects. In these projects we have developed some mobile applications for crowdsourcing and crowdtasking, which got me interested in the factors which govern the behaviour of the volunteers. For now I'm developing this as a hobby, but future versions of this model may be directly related to and used in the DRIVER project.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.1.0
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
VIEW
8
78
648
718
0
0
0
1
1
1
1
1
0
1
1
1
0
50
0
50

INPUTBOX
665
282
746
342
x1
0
1
0
Number

INPUTBOX
665
375
747
435
x2
0
1
0
Number

INPUTBOX
751
281
827
341
y1
0
1
0
Number

INPUTBOX
755
374
834
434
y2
0
1
0
Number

TEXTBOX
670
255
820
273
Recruiting area (x1,y1)
13
0.0
1

CHOOSER
531
21
655
66
Task-type
Task-type
"resolve-issue"
0

BUTTON
414
11
521
69
Submit
NIL
NIL
1
T
OBSERVER
NIL
NIL

BUTTON
196
27
297
60
Next-Issue
NIL
NIL
1
T
OBSERVER
NIL
N

MONITOR
121
17
188
66
Issue-ID
NIL
0
1

BUTTON
12
26
113
59
Prev-Issue
NIL
NIL
1
T
OBSERVER
NIL
P

INPUTBOX
316
10
398
70
Nr-confirmations
0
1
0
Number

TEXTBOX
667
352
817
370
recruiting area (x2,y2)
13
0.0
1

SWITCH
672
30
822
63
Task-nearest?
Task-nearest?
1
1
-1000

TEXTBOX
676
76
826
220
If Task-nearest? switch is on, the tasking area will be ignored and the nearest free volunteers will be tasked.\n\nElse the volunteers will be recruited from the tasking area.
13
0.0
1

@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
